<?php
// Extended functionality for Findexa AI Post Manager
if ( ! defined( 'ABSPATH' ) ) exit;

// NOTE: Almost all REST API endpoints have been removed. 
// WordPress now operates as a client to FastAPI, not as a server.
// Only read-only endpoints remain for Python script compatibility.

// SAFE READ-ONLY REST API ENDPOINTS (for Python script)

add_action( 'rest_api_init', 'rpm_register_safe_readonly_routes' );

function rpm_register_safe_readonly_routes() {
	// Content profile endpoint - read-only, no authentication needed
	register_rest_route( 'rpm/v1', '/content-profile', array(
		'methods'             => 'GET',
		'callback'            => 'rpm_get_content_profile',
		'permission_callback' => '__return_true', // Public read-only
	) );
	
	// Categories endpoint - read-only, no authentication needed
	register_rest_route( 'rpm/v1', '/categories', array(
		'methods'             => 'GET',
		'callback'            => 'rpm_get_categories',
		'permission_callback' => '__return_true', // Public read-only
	) );
	
	// Topics endpoint - POST only for adding topics to queue
	register_rest_route( 'rpm/v1', '/topics', array(
		'methods'             => 'POST',
		'callback'            => 'rpm_add_topics',
		'permission_callback' => '__return_true', // No auth needed (internal use)
	) );
}

function rpm_get_content_profile() {
	return new WP_REST_Response(
		array(
			'success' => true,
			'profile' => array(
				'site_description'   => get_option( 'rpm_site_description', '' ),
				'industry'           => get_option( 'rpm_industry', '' ),
				'target_audience'    => get_option( 'rpm_target_audience', '' ),
				'content_tone'       => get_option( 'rpm_content_tone', 'professional' ),
				'writing_style'      => get_option( 'rpm_writing_style', 'balanced' ),
				'content_length'     => get_option( 'rpm_content_length', 'medium' ),
				'content_goal'       => get_option( 'rpm_content_goal', 'inform' ),
				'expertise_level'    => get_option( 'rpm_expertise_level', 'intermediate' ),
				'brand_keywords'     => get_option( 'rpm_brand_keywords', '' ),
				'avoid_words'        => get_option( 'rpm_avoid_words', '' ),
				'image_search'       => get_option( 'rpm_image_search', '' ),
				'default_status'     => get_option( 'rpm_default_status', 'pending' ),
				'default_category'   => get_option( 'rpm_default_category', '' ),
				'marketing_mode'     => get_option( 'rpm_marketing_mode', false ) ? true : false,
			),
		),
		200
	);
}

function rpm_get_categories() {
	$categories = get_categories(
		array(
			'hide_empty' => false,
			'orderby'    => 'name',
			'order'      => 'ASC',
		)
	);

	$formatted_categories = array();
	foreach ( $categories as $category ) {
		$formatted_categories[] = array(
			'id'          => $category->term_id,
			'name'        => $category->name,
			'slug'        => $category->slug,
			'description' => $category->description,
			'count'       => $category->count,
		);
	}

	return new WP_REST_Response(
		array(
			'success'    => true,
			'categories' => $formatted_categories,
		),
		200
	);
}

function rpm_add_topics( $request ) {
	$params = $request->get_json_params();
	
	if ( empty( $params['topics'] ) || ! is_array( $params['topics'] ) ) {
		return new WP_Error( 'invalid_topics', 'Topics array is required', array( 'status' => 400 ) );
	}
	
	// Get current topic queue
	$topic_queue = get_option( 'rpm_topic_queue', array() );
	
	// Add new topics to queue
	$added_count = 0;
	foreach ( $params['topics'] as $topic ) {
		if ( empty( $topic['subject'] ) ) {
			continue;
		}
		
		// Generate unique ID if not provided
		if ( empty( $topic['id'] ) ) {
			$topic['id'] = uniqid( 'topic_', true );
		}
		
		$topic_queue[] = $topic;
		$added_count++;
	}
	
	// Save updated queue
	update_option( 'rpm_topic_queue', $topic_queue );
	
	return new WP_REST_Response(
		array(
			'success' => true,
			'message' => "Added {$added_count} topics to queue",
			'total_topics' => count( $topic_queue ),
		),
		200
	);
}

// HELPER FUNCTIONS

/**
 * Get all used Pexels IDs from existing content
 * Used for duplicate prevention when selecting images
 */
function rpm_get_used_pexels_ids() {
	global $wpdb;
	
	$used_ids = array();
	
	// Query 1: Get Pexels IDs from post meta (_rpm_pexels_id)
	$post_ids = $wpdb->get_col(
		"SELECT DISTINCT meta_value 
		FROM {$wpdb->postmeta} 
		WHERE meta_key = '_rpm_pexels_id' 
		AND meta_value != ''"
	);
	
	if ( $post_ids ) {
		$used_ids = array_merge( $used_ids, $post_ids );
	}
	
	// Query 2: Get Pexels IDs from post meta (_featured_image_pexels_id)
	$featured_ids = $wpdb->get_col(
		"SELECT DISTINCT meta_value 
		FROM {$wpdb->postmeta} 
		WHERE meta_key = '_featured_image_pexels_id' 
		AND meta_value != ''"
	);
	
	if ( $featured_ids ) {
		$used_ids = array_merge( $used_ids, $featured_ids );
	}
	
	// Query 3: Get Pexels IDs from attachment meta (_pexels_id)
	$attachment_ids = $wpdb->get_col(
		"SELECT DISTINCT meta_value 
		FROM {$wpdb->postmeta} 
		WHERE meta_key = '_pexels_id' 
		AND meta_value != ''"
	);
	
	if ( $attachment_ids ) {
		$used_ids = array_merge( $used_ids, $attachment_ids );
	}
	
	// Query 4: Get from quick-access option cache
	$cached_ids = get_option( 'rpm_used_pexels_ids', array() );
	if ( is_array( $cached_ids ) && ! empty( $cached_ids ) ) {
		$used_ids = array_merge( $used_ids, $cached_ids );
	}
	
	// Remove duplicates and convert to strings
	$used_ids = array_unique( $used_ids );
	$used_ids = array_map( 'strval', $used_ids );
	$used_ids = array_filter( $used_ids ); // Remove empty values
	
	return array_values( $used_ids );
}

/**
 * Track a Pexels image ID as used
 * Stores in a global option for quick duplicate prevention
 * 
 * @param int|string $pexels_id Pexels photo ID to mark as used
 */
function rpm_add_used_pexels_id( $pexels_id ) {
	if ( empty( $pexels_id ) ) {
		return;
	}
	
	// Get current list
	$used_ids = get_option( 'rpm_used_pexels_ids', array() );
	
	if ( ! is_array( $used_ids ) ) {
		$used_ids = array();
	}
	
	// Add new ID (as string)
	$pexels_id = strval( $pexels_id );
	if ( ! in_array( $pexels_id, $used_ids, true ) ) {
		$used_ids[] = $pexels_id;
		
		// Keep only last 200 IDs to avoid option getting too large
		if ( count( $used_ids ) > 200 ) {
			$used_ids = array_slice( $used_ids, -200 );
		}
		
		update_option( 'rpm_used_pexels_ids', $used_ids, false );
	}
}

// ADMIN MENU SETUP

add_action( 'admin_menu', 'rpm_add_extended_menu', 15 );
function rpm_add_extended_menu() {
	// Remove the original settings page if it exists
	remove_submenu_page( 'options-general.php', 'remote-post-manager' );
	
	// Add main menu page - FINDEXA AI Posts (default to Account page)
	add_menu_page(
		__( 'FINDEXA AI Posts', 'remote-post-manager' ),
		__( 'FINDEXA AI Posts', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager',
		'rpm_render_settings_page',
		'dashicons-rss',
		25.5
	);
	
	// Add submenus in desired order: Subscription, Settings, Post Queue, Posts, Tags, Categories
	add_submenu_page(
		'remote-post-manager',
		__( 'Subscription', 'remote-post-manager' ),
		__( 'Subscription', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager',
		'rpm_render_settings_page'
	);
	
	add_submenu_page(
		'remote-post-manager',
		__( 'Content Settings', 'remote-post-manager' ),
		__( 'Content Settings', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager-ai-content',
		'rpm_render_ai_content_page'
	);
	
	// Post Queue menu hidden - posts now generated on-the-fly instead of from queue
	// Keeping code for backward compatibility but removing from UI
	/*
	add_submenu_page(
		'remote-post-manager',
		__( 'Post Queue', 'remote-post-manager' ),
		__( 'Post Queue', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager-post-queue',
		'rpm_render_topic_queue_page'
	);
	*/
	
	add_submenu_page(
		'remote-post-manager',
		__( 'Posts', 'remote-post-manager' ),
		__( 'Posts', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager-auto-posts',
		'rpm_render_auto_posts_page'
	);
	
	add_submenu_page(
		'remote-post-manager',
		__( 'Tags', 'remote-post-manager' ),
		__( 'Tags', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager-tags',
		'rpm_render_tags_page'
	);
	
	add_submenu_page(
		'remote-post-manager',
		__( 'Categories', 'remote-post-manager' ),
		__( 'Categories', 'remote-post-manager' ),
		'manage_options',
		'remote-post-manager-categories',
		'rpm_render_categories_page'
	);
}

// Register all content profile settings
add_action( 'admin_init', 'rpm_register_extended_settings', 20 );
function rpm_register_extended_settings() {
	// AI Content settings
	$ai_settings = array(
		'rpm_site_description' => 'sanitize_textarea_field',
		'rpm_target_audience' => 'sanitize_textarea_field',
		'rpm_industry' => 'sanitize_text_field',
		'rpm_content_tone' => 'sanitize_text_field',
		'rpm_writing_style' => 'sanitize_text_field',
		'rpm_content_goal' => 'sanitize_text_field',
		'rpm_expertise_level' => 'sanitize_text_field',
		'rpm_brand_keywords' => 'sanitize_textarea_field',
		'rpm_avoid_words' => 'sanitize_textarea_field',
		'rpm_content_length' => 'sanitize_text_field',
		'rpm_image_search' => 'sanitize_text_field',
		'rpm_default_status' => 'sanitize_text_field',
		'rpm_default_category' => 'absint',
		'rpm_marketing_mode' => 'rest_sanitize_boolean'
	);
	
	foreach ( $ai_settings as $option => $callback ) {
		register_setting( 'rpm_ai_content_group', $option, array(
			'sanitize_callback' => $callback
		));
	}
}

function rpm_render_topic_queue_page() {
	require_once RPM_PLUGIN_DIR . 'pages/topic-queue.php';
}

function rpm_render_ai_content_page() {
	require_once RPM_PLUGIN_DIR . 'pages/ai-content.php';
}

function rpm_render_settings_page() {
	require_once RPM_PLUGIN_DIR . 'pages/settings.php';
}

function rpm_render_auto_posts_page() {
	require_once RPM_PLUGIN_DIR . 'pages/auto-posts.php';
}
function rpm_render_tags_page() {
	require_once RPM_PLUGIN_DIR . 'pages/tags.php';
}

function rpm_render_categories_page() {
	require_once RPM_PLUGIN_DIR . 'pages/categories.php';
}


// Hook into post creation to save metadata
add_filter( 'rpm_after_post_create', 'rpm_save_auto_post_metadata', 10, 2 );
function rpm_save_auto_post_metadata( $post_id, $params ) {
	// Mark as auto-posted
	if ( isset( $params['auto_posted'] ) && $params['auto_posted'] ) {
		update_post_meta( $post_id, '_rpm_auto_posted', '1' );
	}
	
	// Save metadata
	if ( isset( $params['metadata'] ) && is_array( $params['metadata'] ) ) {
		if ( ! empty( $params['metadata']['subject'] ) ) {
			update_post_meta( $post_id, '_rpm_subject', sanitize_text_field( $params['metadata']['subject'] ) );
		}
		if ( ! empty( $params['metadata']['direction'] ) ) {
			update_post_meta( $post_id, '_rpm_direction', sanitize_textarea_field( $params['metadata']['direction'] ) );
		}
		if ( ! empty( $params['metadata']['image_query'] ) ) {
			update_post_meta( $post_id, '_rpm_image_query', sanitize_text_field( $params['metadata']['image_query'] ) );
		}
		if ( ! empty( $params['metadata']['generated_at'] ) ) {
			update_post_meta( $post_id, '_rpm_generated_at', sanitize_text_field( $params['metadata']['generated_at'] ) );
		}
	}
	
	// Save alternative images if provided
	if ( ! empty( $params['alternative_images'] ) && is_array( $params['alternative_images'] ) ) {
		update_post_meta( $post_id, '_rpm_image_options', $params['alternative_images'] );
	}
	
	return $post_id;
}

// Add meta box to post edit screen for alternative images
add_action( 'add_meta_boxes', 'rpm_add_alternative_images_meta_box' );
function rpm_add_alternative_images_meta_box() {
	add_meta_box(
		'rpm_alternative_images',
		__( 'Alternative Featured Images', 'remote-post-manager' ),
		'rpm_render_alternative_images_meta_box',
		'post',
		'side',
		'default'
	);
}

function rpm_render_alternative_images_meta_box( $post ) {
	// Check if this is an auto-generated post
	$is_auto_posted = get_post_meta( $post->ID, '_rpm_auto_posted', true );
	
	if ( ! $is_auto_posted ) {
		echo '<p>This feature is only available for auto-generated posts.</p>';
		echo '<p style="color: #666; font-size: 11px;"><em>Post ID: ' . $post->ID . ' | Auto-posted: No</em></p>';
		return;
	}
	
	// Get alternative images
	$alternative_images = get_post_meta( $post->ID, '_rpm_image_options', true );
	
	if ( empty( $alternative_images ) || ! is_array( $alternative_images ) ) {
		echo '<p>No alternative images available.</p>';
		echo '<p style="color: #666; font-size: 11px;"><em>Post ID: ' . $post->ID . ' | Auto-posted: Yes | Images: 0</em></p>';
		return;
	}
	
	// Use the proper image picker modal from includes/image-picker-modal.php
	echo '<p>' . count( $alternative_images ) . ' alternative images available</p>';
	echo '<button type="button" class="button button-primary button-large" id="rpm-open-image-picker" style="width: 100%;">';
	echo '🖼️ View ' . count( $alternative_images ) . ' AI-Selected Images';
	echo '</button>';
	echo '<p style="color: #666; font-size: 11px; margin-top: 10px;"><em>Post ID: ' . $post->ID . ' | Images: ' . count( $alternative_images ) . '</em></p>';
	?>
	<script>
	console.log('RPM Meta Box: Rendered for post <?php echo $post->ID; ?> with <?php echo count( $alternative_images ); ?> images');
	console.log('RPM Meta Box: Button ID #rpm-open-image-picker should be present');
	</script>
	<?php
}

// AJAX handler for updating topic
add_action( 'wp_ajax_rpm_update_topic', 'rpm_ajax_update_topic' );
function rpm_ajax_update_topic() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'rpm_update_topic' ) ) {
		wp_send_json_error( 'Invalid nonce' );
	}
	
	// Check permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}
	
	$updated_topic = isset( $_POST['topic'] ) ? $_POST['topic'] : array();
	
	if ( empty( $updated_topic['id'] ) ) {
		wp_send_json_error( 'Missing topic ID' );
	}
	
	$topic_id = $updated_topic['id'];
	$topics = get_option( 'rpm_topic_queue', array() );
	
	// Find and update the topic
	$found = false;
	foreach ( $topics as &$topic ) {
		if ( $topic['id'] === $topic_id ) {
			// Update basic topic fields
			$topic['subject'] = sanitize_text_field( $updated_topic['subject'] );
			$topic['direction'] = sanitize_textarea_field( $updated_topic['direction'] );
			if ( ! empty( $updated_topic['category_id'] ) ) {
				$topic['category_id'] = absint( $updated_topic['category_id'] );
			} else {
				$topic['category_id'] = null;
			}
			
			// Update content profile if provided
			if ( ! empty( $updated_topic['content_profile'] ) ) {
				$profile = $updated_topic['content_profile'];
				
				// Initialize content_profile if it doesn't exist
				if ( ! isset( $topic['content_profile'] ) ) {
					$topic['content_profile'] = array();
				}
				
				// Update each profile field (site-level)
				$topic['content_profile']['site_description'] = sanitize_textarea_field( $profile['site_description'] ?? '' );
				$topic['content_profile']['target_audience'] = sanitize_text_field( $profile['target_audience'] ?? '' );
				$topic['content_profile']['industry'] = sanitize_text_field( $profile['industry'] ?? '' );
				
				// Update each profile field (content-level)
				$topic['content_profile']['content_tone'] = sanitize_text_field( $profile['content_tone'] ?? '' );
				$topic['content_profile']['writing_style'] = sanitize_text_field( $profile['writing_style'] ?? '' );
				$topic['content_profile']['content_goal'] = sanitize_text_field( $profile['content_goal'] ?? '' );
				$topic['content_profile']['expertise_level'] = sanitize_text_field( $profile['expertise_level'] ?? '' );
				$topic['content_profile']['content_length'] = sanitize_text_field( $profile['content_length'] ?? 'medium' );
				$topic['content_profile']['brand_keywords'] = sanitize_textarea_field( $profile['brand_keywords'] ?? '' );
				$topic['content_profile']['avoid_words'] = sanitize_textarea_field( $profile['avoid_words'] ?? '' );
				$topic['content_profile']['image_search'] = sanitize_text_field( $profile['image_search'] ?? '' );
				$topic['content_profile']['marketing_mode'] = (bool) ( $profile['marketing_mode'] ?? false );
				$topic['content_profile']['default_status'] = sanitize_text_field( $profile['default_status'] ?? 'pending' );
			}
			
			$found = true;
			break;
		}
	}
	
	if ( ! $found ) {
		wp_send_json_error( 'Topic not found' );
	}
	
	update_option( 'rpm_topic_queue', $topics );
	wp_send_json_success( 'Topic updated successfully' );
}

// AJAX handler for bulk updating topics
add_action( 'wp_ajax_rpm_bulk_update_topics', 'rpm_ajax_bulk_update_topics' );
function rpm_ajax_bulk_update_topics() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'rpm_bulk_update_topics' ) ) {
		wp_send_json_error( 'Invalid nonce' );
	}
	
	// Check permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}
	
	$topic_ids = isset( $_POST['topic_ids'] ) ? $_POST['topic_ids'] : array();
	$updates = isset( $_POST['updates'] ) ? $_POST['updates'] : array();
	
	if ( empty( $topic_ids ) || ! is_array( $topic_ids ) ) {
		wp_send_json_error( 'No topics selected' );
	}
	
	if ( empty( $updates ) || ! is_array( $updates ) ) {
		wp_send_json_error( 'No updates provided' );
	}
	
	$topics = get_option( 'rpm_topic_queue', array() );
	$updated_count = 0;
	
	// Update each matching topic
	foreach ( $topics as &$topic ) {
		if ( in_array( $topic['id'], $topic_ids, true ) ) {
			// Update category if provided
			if ( isset( $updates['category_id'] ) ) {
				$topic['category_id'] = absint( $updates['category_id'] );
			}
			
			// Update content profile if provided
			if ( isset( $updates['content_profile'] ) && is_array( $updates['content_profile'] ) ) {
				// Initialize content_profile if it doesn't exist
				if ( ! isset( $topic['content_profile'] ) ) {
					$topic['content_profile'] = array();
				}
				
				$profile_updates = $updates['content_profile'];
				
				// Update only provided fields
				if ( isset( $profile_updates['content_tone'] ) ) {
					$topic['content_profile']['content_tone'] = sanitize_text_field( $profile_updates['content_tone'] );
				}
				if ( isset( $profile_updates['writing_style'] ) ) {
					$topic['content_profile']['writing_style'] = sanitize_text_field( $profile_updates['writing_style'] );
				}
				if ( isset( $profile_updates['content_goal'] ) ) {
					$topic['content_profile']['content_goal'] = sanitize_text_field( $profile_updates['content_goal'] );
				}
				if ( isset( $profile_updates['expertise_level'] ) ) {
					$topic['content_profile']['expertise_level'] = sanitize_text_field( $profile_updates['expertise_level'] );
				}
				if ( isset( $profile_updates['content_length'] ) ) {
					$topic['content_profile']['content_length'] = sanitize_text_field( $profile_updates['content_length'] );
				}
				if ( isset( $profile_updates['image_search'] ) ) {
					$topic['content_profile']['image_search'] = sanitize_text_field( $profile_updates['image_search'] );
				}
				if ( isset( $profile_updates['brand_keywords'] ) ) {
					$topic['content_profile']['brand_keywords'] = sanitize_textarea_field( $profile_updates['brand_keywords'] );
				}
				if ( isset( $profile_updates['avoid_words'] ) ) {
					$topic['content_profile']['avoid_words'] = sanitize_textarea_field( $profile_updates['avoid_words'] );
				}
				if ( isset( $profile_updates['marketing_mode'] ) ) {
					$topic['content_profile']['marketing_mode'] = (bool) $profile_updates['marketing_mode'];
				}
				if ( isset( $profile_updates['default_status'] ) ) {
					$topic['content_profile']['default_status'] = sanitize_text_field( $profile_updates['default_status'] );
				}
			}
			
			$updated_count++;
		}
	}
	
	if ( $updated_count === 0 ) {
		wp_send_json_error( 'No topics were updated' );
	}
	
	update_option( 'rpm_topic_queue', $topics );
	wp_send_json_success( sprintf( '%d topic(s) updated successfully', $updated_count ) );
}

// AJAX handler for getting alternative images
add_action( 'wp_ajax_rpm_get_alternative_images', 'rpm_ajax_get_alternative_images' );
function rpm_ajax_get_alternative_images() {
	$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
	
	if ( ! $post_id ) {
		wp_send_json_error( 'Missing post ID' );
	}
	
	// Check permissions
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}
	
	// Get alternative images
	$alternative_images = get_post_meta( $post_id, '_rpm_image_options', true );
	
	if ( empty( $alternative_images ) || ! is_array( $alternative_images ) ) {
		wp_send_json_error( 'No alternative images found' );
	}
	
	// Extract URLs from image objects
	$image_urls = array();
	foreach ( $alternative_images as $image ) {
		if ( is_array( $image ) && isset( $image['url'] ) ) {
			$image_urls[] = $image['url'];
		} elseif ( is_string( $image ) ) {
			$image_urls[] = $image;
		}
	}
	
	if ( empty( $image_urls ) ) {
		wp_send_json_error( 'No valid image URLs found' );
	}
	
	// Get current featured image
	$current_thumbnail_id = get_post_thumbnail_id( $post_id );
	$current_thumbnail_url = $current_thumbnail_id ? wp_get_attachment_url( $current_thumbnail_id ) : '';
	
	wp_send_json_success( array(
		'images' => $image_urls,
		'current_image' => $current_thumbnail_url
	));
}

// AJAX handler for adding tags (bulk support)
add_action( 'wp_ajax_rpm_add_tag', 'rpm_ajax_add_tag' );
function rpm_ajax_add_tag() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'rpm_add_tag_action' ) ) {
		wp_send_json_error( 'Invalid nonce' );
	}
	
	// Check permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}
	
	$tag_name = isset( $_POST['tag_name'] ) ? sanitize_text_field( $_POST['tag_name'] ) : '';
	
	if ( empty( $tag_name ) ) {
		wp_send_json_error( 'Tag name is required' );
	}
	
	$result = wp_insert_term( $tag_name, 'post_tag' );
	
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message() );
	}
	
	wp_send_json_success( array(
		'message' => 'Tag added successfully',
		'term_id' => $result['term_id']
	));
}

// AJAX handler for adding categories (bulk support)
add_action( 'wp_ajax_rpm_add_category', 'rpm_ajax_add_category' );
function rpm_ajax_add_category() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'rpm_add_category_action' ) ) {
		wp_send_json_error( 'Invalid nonce' );
	}
	
	// Check permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}
	
	$category_name = isset( $_POST['category_name'] ) ? sanitize_text_field( $_POST['category_name'] ) : '';
	$parent_id = isset( $_POST['parent_id'] ) ? intval( $_POST['parent_id'] ) : 0;
	
	if ( empty( $category_name ) ) {
		wp_send_json_error( 'Category name is required' );
	}
	
	$result = wp_insert_term( $category_name, 'category', array(
		'parent' => $parent_id
	));
	
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message() );
	}
	
	wp_send_json_success( array(
		'message' => 'Category added successfully',
		'term_id' => $result['term_id']
	));
}

// Auto-open featured image selector when "Change Image" link is clicked
add_action( 'admin_footer-post.php', 'rpm_auto_open_featured_image' );
function rpm_auto_open_featured_image() {
	if ( isset( $_GET['open_media'] ) && $_GET['open_media'] == '1' ) {
		?>
		<script type="text/javascript">
		jQuery(document).ready(function($) {
			// Wait for page to fully load
			setTimeout(function() {
				// Click the "Set featured image" link
				var $featuredImageLink = $('#set-post-thumbnail');
				if ($featuredImageLink.length) {
					$featuredImageLink.trigger('click');
				} else {
					// Try alternate selector for newer WordPress versions
					$('#postimagediv .set-post-thumbnail').trigger('click');
				}
			}, 500);
		});
		</script>
		<?php
	}
}

