<?php
/**
 * Plugin Name: Findexa AI Post Manager
 * Plugin URI: https://findexa.ca/plugins/remote-post-manager
 * Description: AI-powered content creation and management system with intelligent image selection, duplicate tracking, and automated posting capabilities.
 * Version: 1.0.3
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Author: Findexa Team
 * Author URI: https://findexa.ca
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: remote-post-manager
 * Domain Path: /languages
 *
 * @package RemotePostManager
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// Allow localhost connections for FastAPI communication
add_filter( 'http_request_host_is_external', function( $is_external, $host ) {
	if ( $host === '127.0.0.1' || $host === 'localhost' ) {
		return true;
	}
	return $is_external;
}, 10, 2 );


// Define plugin constants.
define( 'RPM_VERSION', '1.0.2' );
define( 'RPM_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'RPM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'RPM_PLUGIN_FILE', __FILE__ );

/**
 * Main Plugin Class
 */
class Remote_Post_Manager {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		// Load plugin text domain.
		add_action( 'init', array( $this, 'load_textdomain' ) );

		// Admin menu and settings.
		// NOTE: Menu is now handled by rpm-extended.php
		// add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Add settings link on plugins page.
		add_filter( 'plugin_action_links_' . plugin_basename( RPM_PLUGIN_FILE ), array( $this, 'add_settings_link' ) );

		// Plugin activation/deactivation hooks.
		register_activation_hook( RPM_PLUGIN_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( RPM_PLUGIN_FILE, array( $this, 'deactivate' ) );
	}

	/**
	 * Get instance of this class.
	 *
	 * @return object Instance.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load plugin text domain for translations.
	 */
	public function load_textdomain() {
		load_plugin_textdomain(
			'remote-post-manager',
			false,
			dirname( plugin_basename( RPM_PLUGIN_FILE ) ) . '/languages/'
		);
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_options_page(
			__( 'Findexa AI Post Manager Settings', 'remote-post-manager' ),
			__( 'Findexa AI Post Manager', 'remote-post-manager' ),
			'manage_options',
			'remote-post-manager',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		// General settings
		register_setting(
			'rpm_settings_group',
			'rpm_api_key',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			)
		);

		register_setting(
			'rpm_settings_group',
			'rpm_enable_logging',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => false,
			)
		);

		// Auto posting settings - separate group to prevent form reset
		register_setting(
			'rpm_daily_limit_group',
			'rpm_auto_posting_enabled',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => false,
			)
		);
		
		register_setting(
			'rpm_daily_limit_group',
			'rpm_posts_per_day',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 1,
			)
		);
		
		register_setting(
			'rpm_daily_limit_group',
			'rpm_empty_queue_behavior',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => 'skip',
			)
		);
		
		register_setting(
			'rpm_daily_limit_group',
			'rpm_notify_empty_queue',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => false,
			)
		);

		register_setting(
			'rpm_settings_group',
			'rpm_fastapi_url',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'esc_url_raw',
				'default'           => 'http://localhost:8000',
			)
		);

		register_setting(
			'rpm_settings_group',
			'rpm_fastapi_key',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			)
		);

		// FastAPI settings - separate group to prevent form reset
		register_setting(
			'rpm_fastapi_group',
			'rpm_fastapi_url',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'esc_url_raw',
				'default'           => 'http://localhost:8000',
			)
		);

	register_setting(
		'rpm_fastapi_group',
		'rpm_fastapi_key',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => '',
		)
	);

	add_settings_section(
			'rpm_main_section',
			__( 'API Configuration', 'remote-post-manager' ),
			array( $this, 'render_section_callback' ),
			'remote-post-manager'
		);


		add_settings_field(
			'rpm_api_key',
			__( 'API Key', 'remote-post-manager' ),
			array( $this, 'render_api_key_field' ),
			'remote-post-manager',
			'rpm_main_section'
		);

		add_settings_field(
			'rpm_enable_logging',
			__( 'Enable Logging', 'remote-post-manager' ),
			array( $this, 'render_logging_field' ),
			'remote-post-manager',
			'rpm_main_section'
		);
	}

	/**
	 * Render settings section callback.
	 */
	public function render_section_callback() {
		echo '<p>' . esc_html__( 'Configure the API key for remote post creation. Keep this key secure.', 'remote-post-manager' ) . '</p>';
	}

	/**
	 * Render API key field.
	 */
	public function render_api_key_field() {
		$api_key = get_option( 'rpm_api_key', '' );
		?>
		<input type="text" id="rpm_api_key" name="rpm_api_key" value="<?php echo esc_attr( $api_key ); ?>" class="regular-text" />
		<button type="button" class="button" id="rpm_generate_key"><?php esc_html_e( 'Generate New Key', 'remote-post-manager' ); ?></button>
		<p class="description">
			<?php esc_html_e( 'Use this API key in the Authorization header as: Bearer YOUR_API_KEY', 'remote-post-manager' ); ?>
		</p>
		<script>
		document.getElementById('rpm_generate_key').addEventListener('click', function() {
			var key = Array.from({length: 32}, () => Math.random().toString(36)[2]).join('');
			document.getElementById('rpm_api_key').value = key;
		});
		</script>
		<?php
	}

	/**
	 * Render logging field.
	 */
	public function render_logging_field() {
		$enabled = get_option( 'rpm_enable_logging', false );

		register_setting(
			'rpm_settings_group',
			'rpm_posts_per_day',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 5,
			)
		);
		?>

		register_setting(
			'rpm_settings_group',
			'rpm_posts_per_day',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 5,
			)
		);
		<label>

		register_setting(
			'rpm_settings_group',
			'rpm_posts_per_day',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 5,
			)
		);
			<input type="checkbox" id="rpm_enable_logging" name="rpm_enable_logging" value="1" <?php checked( $enabled, true ); ?> />

		register_setting(
			'rpm_settings_group',
			'rpm_posts_per_day',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 5,
			)
		);
			<?php esc_html_e( 'Enable request logging (for debugging)', 'remote-post-manager' ); ?>
		</label>
		<?php
	}

	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission messages.
		if ( isset( $_GET['settings-updated'] ) ) {
			add_settings_error(
				'rpm_messages',
				'rpm_message',
				__( 'Settings saved successfully.', 'remote-post-manager' ),
				'updated'
			);
		}

		settings_errors( 'rpm_messages' );
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<form action="options.php" method="post">
				<?php
				settings_fields( 'rpm_settings_group' );
				do_settings_sections( 'remote-post-manager' );
				submit_button( __( 'Save Settings', 'remote-post-manager' ) );
				?>
			</form>

			<hr>

			<h2><?php esc_html_e( 'API Documentation', 'remote-post-manager' ); ?></h2>
			
			<h3><?php esc_html_e( 'Endpoints', 'remote-post-manager' ); ?></h3>
			
			<div class="rpm-docs">
				<h4><?php esc_html_e( '1. Create Post', 'remote-post-manager' ); ?></h4>
				<p><strong><?php esc_html_e( 'Endpoint:', 'remote-post-manager' ); ?></strong> <code>POST <?php echo esc_url( rest_url( 'rpm/v1/post' ) ); ?></code></p>
				<p><strong><?php esc_html_e( 'Headers:', 'remote-post-manager' ); ?></strong></p>
				<pre>Authorization: Bearer YOUR_API_KEY
Content-Type: application/json</pre>
				<p><strong><?php esc_html_e( 'Body (JSON):', 'remote-post-manager' ); ?></strong></p>
				<pre>{
  "title": "Post Title",
  "content": "&lt;p&gt;Post content here&lt;/p&gt;",
  "excerpt": "Optional excerpt",
  "status": "publish",
  "categories": [1, 5],
  "tags": ["tag1", "tag2"],
  "featured_image": "https://example.com/image.jpg"
}</pre>

				<h4><?php esc_html_e( '2. Update Post', 'remote-post-manager' ); ?></h4>
				<p><strong><?php esc_html_e( 'Endpoint:', 'remote-post-manager' ); ?></strong> <code>PUT <?php echo esc_url( rest_url( 'rpm/v1/post/{id}' ) ); ?></code></p>
				
				<h4><?php esc_html_e( '3. Delete Post', 'remote-post-manager' ); ?></h4>
				<p><strong><?php esc_html_e( 'Endpoint:', 'remote-post-manager' ); ?></strong> <code>DELETE <?php echo esc_url( rest_url( 'rpm/v1/post/{id}' ) ); ?></code></p>
				
				<h4><?php esc_html_e( '4. Get Categories', 'remote-post-manager' ); ?></h4>
				<p><strong><?php esc_html_e( 'Endpoint:', 'remote-post-manager' ); ?></strong> <code>GET <?php echo esc_url( rest_url( 'rpm/v1/categories' ) ); ?></code></p>
				
				<h4><?php esc_html_e( '5. Get Tags', 'remote-post-manager' ); ?></h4>
				<p><strong><?php esc_html_e( 'Endpoint:', 'remote-post-manager' ); ?></strong> <code>GET <?php echo esc_url( rest_url( 'rpm/v1/tags' ) ); ?></code></p>
			</div>

			<hr>

			<h3><?php esc_html_e( 'Example cURL Commands', 'remote-post-manager' ); ?></h3>
			<pre style="background: #f5f5f5; padding: 15px; border-radius: 5px;">
# Create a post
curl -X POST <?php echo esc_url( rest_url( 'rpm/v1/post' ) ); ?> \
  -H "Authorization: Bearer YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "title": "My New Post",
    "content": "<p>This is the post content</p>",
    "status": "publish",
    "categories": [1],
    "tags": ["news", "update"]
  }'

# Get categories
curl -X GET <?php echo esc_url( rest_url( 'rpm/v1/categories' ) ); ?> \
  -H "Authorization: Bearer YOUR_API_KEY"
</pre>
		</div>

		<style>
		.rpm-docs {
			background: #fff;
			padding: 20px;
			border: 1px solid #ddd;
			border-radius: 5px;
			margin-top: 20px;
		}
		.rpm-docs h4 {
			margin-top: 20px;
			color: #23282d;
		}
		.rpm-docs pre {
			background: #f5f5f5;
			padding: 10px;
			border-radius: 3px;
			overflow-x: auto;
		}
		.rpm-docs code {
			background: #f0f0f0;
			padding: 2px 6px;
			border-radius: 3px;
		}
		</style>
		<?php
	}

	/**
	 * Add settings link on plugins page.
	 *
	 * @param array $links Plugin action links.
	 * @return array Modified links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="' . esc_url( admin_url( 'options-general.php?page=remote-post-manager' ) ) . '">' . __( 'Settings', 'remote-post-manager' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Plugin activation.
	 */
	public function activate() {
		// Generate a default API key if not exists.
		if ( ! get_option( 'rpm_api_key' ) ) {
			$api_key = wp_generate_password( 32, false, false );
			update_option( 'rpm_api_key', $api_key );
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation.
	 */
	public function deactivate() {
		// Flush rewrite rules.
		flush_rewrite_rules();
	}
}

// Load extended features
require_once RPM_PLUGIN_DIR . 'rpm-extended.php';

// Initialize the plugin class.
Remote_Post_Manager::get_instance();

// Initialize the plugin.
// require_once RPM_PLUGIN_DIR . 'image-requirements-addon.php';

// Load image requirements add-on.
// require_once RPM_PLUGIN_DIR . 'image-requirements-addon.php';
require_once RPM_PLUGIN_DIR . "includes/ajax-handlers.php";



// Load image picker modal
require_once RPM_PLUGIN_DIR . "includes/image-picker-modal.php";

// Stripe integration is only loaded on findexa.ca (not in distributed plugin)
// Customer billing is handled through findexa.ca website
if ( file_exists( RPM_PLUGIN_DIR . "includes/stripe-integration.php" ) ) {
    require_once RPM_PLUGIN_DIR . "includes/stripe-integration.php";
}

// Load Schema.org structured data and Open Graph output
require_once RPM_PLUGIN_DIR . "includes/schema-output.php";

// Load media cleanup (auto-delete images when posts are deleted)
require_once RPM_PLUGIN_DIR . "includes/media-cleanup.php";

// Load WP-Cron scheduler for automated daily posting
require_once RPM_PLUGIN_DIR . "includes/cron-scheduler.php";

// Register activation/deactivation hooks for cron
register_activation_hook( __FILE__, 'rpm_cron_activation' );
register_deactivation_hook( __FILE__, 'rpm_cron_deactivation' );
