<?php
/**
 * Used Images Management Page
 * Display and manage tracked Pexels image IDs
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Handle actions
if ( isset( $_POST['rpm_clear_all_images'] ) && check_admin_referer( 'rpm_clear_all_images' ) ) {
	delete_option( 'rpm_used_pexels_ids' );
	echo '<div class="notice notice-success"><p>✓ All tracked images have been cleared.</p></div>';
}

if ( isset( $_POST['rpm_clear_old_images'] ) && check_admin_referer( 'rpm_clear_old_images' ) ) {
	$months = isset( $_POST['months'] ) ? intval( $_POST['months'] ) : 6;
	
	// Get all posts with Pexels IDs
	$args = array(
		'post_type'      => 'post',
		'posts_per_page' => -1,
		'meta_query'     => array(
			array(
				'key'     => '_rpm_pexels_id',
				'compare' => 'EXISTS',
			),
		),
		'date_query'     => array(
			array(
				'after' => "-$months months",
			),
		),
	);
	
	$recent_posts = get_posts( $args );
	$keep_ids = array();
	
	foreach ( $recent_posts as $post ) {
		$pexels_id = get_post_meta( $post->ID, '_rpm_pexels_id', true );
		if ( $pexels_id ) {
			$keep_ids[] = strval( $pexels_id );
		}
	}
	
	// Update the used IDs to only keep recent ones
	update_option( 'rpm_used_pexels_ids', array_unique( $keep_ids ) );
	
	$removed = count( get_option( 'rpm_used_pexels_ids', array() ) ) - count( $keep_ids );
	echo '<div class="notice notice-success"><p>✓ Cleared ' . $removed . ' image IDs older than ' . $months . ' months.</p></div>';
}

// Get used image IDs
$used_ids = get_option( 'rpm_used_pexels_ids', array() );
if ( ! is_array( $used_ids ) ) {
	$used_ids = array();
}

// Get sample of recent images with their post data
$recent_images = array();
$args = array(
	'post_type'      => 'post',
	'posts_per_page' => 20,
	'orderby'        => 'date',
	'order'          => 'DESC',
	'meta_query'     => array(
		array(
			'key'     => '_rpm_pexels_id',
			'compare' => 'EXISTS',
		),
	),
);

$posts_with_images = get_posts( $args );
foreach ( $posts_with_images as $post ) {
	$pexels_id = get_post_meta( $post->ID, '_rpm_pexels_id', true );
	$thumbnail_id = get_post_thumbnail_id( $post->ID );
	$thumbnail_url = $thumbnail_id ? wp_get_attachment_url( $thumbnail_id ) : '';
	
	$recent_images[] = array(
		'pexels_id'     => $pexels_id,
		'post_title'    => $post->post_title,
		'post_url'      => get_permalink( $post->ID ),
		'thumbnail_url' => $thumbnail_url,
		'date'          => get_the_date( '', $post ),
	);
}

?>

<div class="wrap">
	<h1>🖼️ Used Pexels Images</h1>
	<p>Track and manage Pexels images that have been used to avoid duplicates across posts.</p>

	<div class="rpm-stats-cards" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0;">
		<div class="rpm-stat-card" style="background: white; padding: 20px; border-radius: 8px; border-left: 4px solid #2271b1; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
			<div style="font-size: 14px; color: #666; margin-bottom: 8px;">Total Tracked Images</div>
			<div style="font-size: 32px; font-weight: 600; color: #2271b1;"><?php echo number_format( count( $used_ids ) ); ?></div>
		</div>
		
		<div class="rpm-stat-card" style="background: white; padding: 20px; border-radius: 8px; border-left: 4px solid #00a32a; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
			<div style="font-size: 14px; color: #666; margin-bottom: 8px;">Posts with Tracked Images</div>
			<div style="font-size: 32px; font-weight: 600; color: #00a32a;"><?php echo number_format( count( $recent_images ) ); ?>+</div>
		</div>
		
		<div class="rpm-stat-card" style="background: white; padding: 20px; border-radius: 8px; border-left: 4px solid #d63638; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
			<div style="font-size: 14px; color: #666; margin-bottom: 8px;">Storage Used</div>
			<div style="font-size: 32px; font-weight: 600; color: #d63638;"><?php echo number_format( strlen( serialize( $used_ids ) ) / 1024, 1 ); ?> KB</div>
		</div>
	</div>

	<!-- Management Actions -->
	<div class="card" style="max-width: 800px; margin-top: 30px;">
		<h2 class="title">Management Actions</h2>
		
		<div style="padding: 20px;">
			<h3>Clear All Tracked Images</h3>
			<p>This will remove all tracked Pexels IDs. Images may be reused in future posts.</p>
			<form method="post" onsubmit="return confirm('Are you sure you want to clear ALL tracked images? This cannot be undone.');">
				<?php wp_nonce_field( 'rpm_clear_all_images' ); ?>
				<button type="submit" name="rpm_clear_all_images" class="button button-secondary">🗑️ Clear All Images</button>
			</form>

			<hr style="margin: 30px 0;">

			<h3>Clear Old Tracked Images</h3>
			<p>Remove image IDs from posts older than a specific number of months. This allows reusing older images.</p>
			<form method="post" onsubmit="return confirm('Are you sure you want to clear old image IDs?');">
				<?php wp_nonce_field( 'rpm_clear_old_images' ); ?>
				<label for="months">Clear images older than:</label>
				<select name="months" id="months">
					<option value="3">3 months</option>
					<option value="6" selected>6 months</option>
					<option value="12">12 months</option>
					<option value="24">24 months</option>
				</select>
				<button type="submit" name="rpm_clear_old_images" class="button button-secondary">🧹 Clear Old Images</button>
			</form>
		</div>
	</div>

	<!-- Recent Images -->
	<?php if ( ! empty( $recent_images ) ) : ?>
	<div class="card" style="margin-top: 30px;">
		<h2 class="title">Recent Images (Last 20)</h2>
		<div style="padding: 20px;">
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<th style="width: 80px;">Thumbnail</th>
						<th>Post Title</th>
						<th style="width: 120px;">Pexels ID</th>
						<th style="width: 120px;">Date</th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $recent_images as $img ) : ?>
					<tr>
						<td>
							<?php if ( $img['thumbnail_url'] ) : ?>
								<img src="<?php echo esc_url( $img['thumbnail_url'] ); ?>" 
								     style="width: 60px; height: 60px; object-fit: cover; border-radius: 4px;" 
								     alt="Thumbnail">
							<?php else : ?>
								<div style="width: 60px; height: 60px; background: #f0f0f1; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #999;">📷</div>
							<?php endif; ?>
						</td>
						<td>
							<a href="<?php echo esc_url( $img['post_url'] ); ?>" target="_blank">
								<?php echo esc_html( $img['post_title'] ); ?>
							</a>
						</td>
						<td>
							<code><?php echo esc_html( $img['pexels_id'] ); ?></code>
						</td>
						<td><?php echo esc_html( $img['date'] ); ?></td>
					</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		</div>
	</div>
	<?php else : ?>
	<div class="notice notice-info" style="margin-top: 30px;">
		<p>No posts with tracked Pexels images found.</p>
	</div>
	<?php endif; ?>

	<!-- Sample of Tracked IDs -->
	<?php if ( ! empty( $used_ids ) ) : ?>
	<div class="card" style="margin-top: 30px;">
		<h2 class="title">Sample of Tracked IDs (First 50)</h2>
		<div style="padding: 20px;">
			<div style="font-family: monospace; background: #f0f0f1; padding: 15px; border-radius: 4px; max-height: 200px; overflow-y: auto;">
				<?php 
				$sample = array_slice( $used_ids, 0, 50 );
				echo implode( ', ', array_map( 'esc_html', $sample ) ); 
				if ( count( $used_ids ) > 50 ) {
					echo '<br><em>... and ' . ( count( $used_ids ) - 50 ) . ' more</em>';
				}
				?>
			</div>
		</div>
	</div>
	<?php endif; ?>

	<!-- Documentation -->
	<div class="card" style="margin-top: 30px; margin-bottom: 30px;">
		<h2 class="title">How It Works</h2>
		<div style="padding: 20px;">
			<ol style="line-height: 1.8;">
				<li><strong>Automatic Tracking:</strong> When the Python script creates a post with a Pexels image, it automatically records the Pexels ID.</li>
				<li><strong>Duplicate Prevention:</strong> Before searching for images, the script fetches the list of used IDs and filters them out.</li>
				<li><strong>Pagination:</strong> If not enough unique images are found on the first page, the script automatically searches additional pages.</li>
				<li><strong>Metadata Storage:</strong> Pexels IDs are stored in both post metadata (<code>_rpm_pexels_id</code>) and attachment metadata (<code>_pexels_id</code>).</li>
				<li><strong>Cleanup Options:</strong> You can clear all tracked IDs or just remove IDs from old posts to allow reusing vintage images.</li>
			</ol>
			
			<h3 style="margin-top: 20px;">API Endpoints</h3>
			<p style="font-size: 12px; color: #646970; margin: 5px 0;">Findexa AI Post Manager REST API endpoints for image tracking:</p>
			<ul style="line-height: 1.8;">
				<li><code>GET /wp-json/rpm/v1/used-images</code> - Fetch list of used Pexels IDs</li>
				<li><code>POST /wp-json/rpm/v1/used-images</code> - Add a new used Pexels ID</li>
			</ul>
		</div>
	</div>
</div>

<style>
.rpm-stats-cards .rpm-stat-card {
	transition: transform 0.2s, box-shadow 0.2s;
}
.rpm-stats-cards .rpm-stat-card:hover {
	transform: translateY(-2px);
	box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}
</style>
