<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// Handle tag actions
$message = '';
$message_type = '';

if ( isset( $_POST['rpm_add_tag'] ) && check_admin_referer( 'rpm_add_tag_action', 'rpm_tag_nonce' ) ) {
	$tag_name = sanitize_text_field( $_POST['tag_name'] );
	if ( ! empty( $tag_name ) ) {
		$result = wp_insert_term( $tag_name, 'post_tag' );
		if ( ! is_wp_error( $result ) ) {
			$message = 'Tag "' . esc_html( $tag_name ) . '" added successfully!';
			$message_type = 'updated';
		} else {
			$message = 'Error: ' . $result->get_error_message();
			$message_type = 'error';
		}
	} else {
		$message = 'Please enter a tag name.';
		$message_type = 'error';
	}
}

if ( isset( $_POST['rpm_delete_tag'] ) && check_admin_referer( 'rpm_delete_tag_action', 'rpm_delete_tag_nonce' ) ) {
	$tag_id = intval( $_POST['tag_id'] );
	$result = wp_delete_term( $tag_id, 'post_tag' );
	if ( ! is_wp_error( $result ) && $result ) {
		$message = 'Tag deleted successfully!';
		$message_type = 'updated';
	} else {
		$message = 'Error deleting tag.';
		$message_type = 'error';
	}
}

// Get all tags
$args = array(
	'taxonomy' => 'post_tag',
	'hide_empty' => false,
	'orderby' => 'name',
	'order' => 'ASC'
);
$all_tags = get_terms( $args );
$tag_count = count( $all_tags );
?>

<div class="wrap">
	<h1 class="wp-heading-inline">
		<span class="dashicons dashicons-tag" style="font-size: 28px; width: 28px; height: 28px; vertical-align: middle;"></span>
		Tags Manager
	</h1>
	<p class="description" style="margin-top: 10px; font-size: 14px;">Manage WordPress tags used by AI-generated posts. View, edit, and delete tags from your site.</p>
	<hr class="wp-header-end">
	
	<?php if ( ! empty( $message ) ): ?>
		<div class="notice notice-<?php echo esc_attr( $message_type ); ?> is-dismissible">
			<p><?php echo esc_html( $message ); ?></p>
		</div>
	<?php endif; ?>
	
	<div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<h2 style="margin-top: 0;">
			<span class="dashicons dashicons-plus-alt" style="font-size: 20px; width: 20px; height: 20px; vertical-align: middle;"></span>
			Add New Tag
		</h2>
		<p class="description">Enter a tag name to create a new tag. You can add multiple tags by separating them with commas.</p>
		
		<form method="post" action="" id="rpm-add-tag-form" style="margin-top: 20px;">
			<?php wp_nonce_field( 'rpm_add_tag_action', 'rpm_tag_nonce' ); ?>
			
			<div style="display: flex; gap: 10px; align-items: flex-start; flex-wrap: wrap;">
				<div style="flex: 1; min-width: 300px;">
					<input 
						type="text" 
						name="tag_name" 
						id="rpm_tag_name" 
						class="regular-text" 
						placeholder="Enter tag name (e.g., Technology, News, Tutorial)"
						style="width: 100%; font-size: 14px; padding: 8px;"
					/>
					<p class="description" style="margin-top: 5px;">
						💡 Tip: Separate multiple tags with commas (e.g., "tech, innovation, startup")
					</p>
				</div>
				<button type="submit" name="rpm_add_tag" class="button button-primary" style="height: 40px; padding: 0 20px;">
					<span class="dashicons dashicons-plus-alt" style="vertical-align: middle; margin-top: 8px;"></span>
					Add Tag
				</button>
			</div>
		</form>
		
		<div id="rpm-bulk-add-result" style="margin-top: 15px;"></div>
	</div>
	
	<!-- AI-Powered Tag Suggestions -->
	<div class="ai-powered-suggestions" style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<h2 style="margin-top: 0;">
			<span class="dashicons dashicons-lightbulb" style="font-size: 20px; color: #f0b849; vertical-align: middle;"></span>
			🤖 AI-Powered Tag Suggestions
		</h2>
		<p class="description">
			Let AI analyze your site's content and suggest relevant tags. Select pages to analyze and specify how many tags you'd like.
		</p>
		
		<!-- Info Banner -->
		<div style="background: #f0f6fc; border-left: 3px solid #2271b1; padding: 15px; margin: 20px 0; border-radius: 4px;">
			<strong>💡 How it works:</strong>
			<ol style="margin: 10px 0 0 20px; padding: 0;">
				<li>Select pages that represent your site's core content</li>
				<li>Choose how many tags you want (5-50 tags)</li>
				<li>Click "Generate Suggestions" and wait 20-30 seconds</li>
				<li>Review AI suggestions and select which tags to create</li>
			</ol>
		</div>
		
		<!-- Page Selector -->
		<div style="margin: 20px 0;">
			<h3 style="margin-bottom: 10px;">📄 Select Pages to Analyze:</h3>
			<div style="max-height: 300px; overflow-y: auto; border: 1px solid #c3c4c7; border-radius: 4px; padding: 15px; background: #f9f9f9;">
				<?php
				$pages = get_pages(['post_status' => 'publish', 'number' => 100, 'sort_column' => 'post_modified']);
				$homepage_id = get_option('page_on_front');
				
				if (empty($pages)) {
					echo '<p style="color: #666; font-style: italic;">No published pages found.</p>';
				} else {
					foreach ($pages as $page) {
						$is_homepage = ($page->ID == $homepage_id);
						$checked = $is_homepage ? 'checked' : '';
						echo '<div style="margin-bottom: 8px;">';
						echo '<label style="display: flex; align-items: center; gap: 8px;">';
						echo '<input type="checkbox" name="rpm_suggest_pages[]" value="' . esc_attr($page->ID) . '" ' . $checked . ' />';
						echo '<span>' . esc_html($page->post_title);
						if ($is_homepage) echo ' <span style="background: #00a32a; color: white; padding: 2px 6px; border-radius: 3px; font-size: 10px; margin-left: 5px;">HOME</span>';
						echo '</span>';
						echo '</label>';
						echo '</div>';
					}
				}
				?>
			</div>
		</div>
		
		<!-- Count Input & Generate Button -->
		<div style="display: flex; gap: 15px; align-items: flex-end; margin-top: 20px;">
			<div style="flex: 1;">
				<label for="rpm_suggest_tags_count" style="display: block; margin-bottom: 5px; font-weight: 600;">
					🔢 How many tags to suggest?
				</label>
				<input 
					type="number" 
					id="rpm_suggest_tags_count" 
					name="rpm_suggest_tags_count" 
					value="10" 
					min="5" 
					max="50" 
					class="regular-text"
					style="width: 150px;"
				/>
				<p class="description" style="margin-top: 5px;">Between 5 and 50 tags</p>
			</div>
			
			<button type="button" id="rpm_generate_tag_suggestions" class="button button-primary" style="height: 40px; padding: 0 20px;">
				<span class="dashicons dashicons-lightbulb" style="vertical-align: middle; margin-top: 8px;"></span>
				🤖 Generate Tag Suggestions
			</button>
		</div>
		
		<!-- Status Area -->
		<div id="rpm_tag_suggestions_status" style="margin-top: 20px;"></div>
		
		<!-- Results Area (initially hidden) -->
		<div id="rpm_tag_suggestions_results" style="display: none; margin-top: 20px;">
			<h3 style="margin-bottom: 15px;">✨ AI Suggested Tags:</h3>
			<p class="description" style="margin-bottom: 15px;">
				Select the tags you want to create, then click "Create Selected Tags" below.
			</p>
			
			<div id="rpm_tag_suggestions_list" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px; margin-bottom: 20px;">
				<!-- Tags will be dynamically inserted here -->
			</div>
			
			<div style="display: flex; gap: 10px;">
				<button type="button" id="rpm_create_selected_tags" class="button button-primary">
					<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-top: 4px;"></span>
					Create Selected Tags
				</button>
				<button type="button" id="rpm_select_all_tags" class="button">Select All</button>
				<button type="button" id="rpm_deselect_all_tags" class="button">Deselect All</button>
			</div>
		</div>
	</div>
	
	<div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
			<h2 style="margin: 0;">
				<span class="dashicons dashicons-list-view" style="font-size: 20px; width: 20px; height: 20px; vertical-align: middle;"></span>
				All Tags (<?php echo number_format_i18n( $tag_count ); ?>)
			</h2>
			
			<div style="display: flex; gap: 10px; align-items: center;">
				<input 
					type="text" 
					id="rpm-tag-search" 
					placeholder="🔍 Search tags..." 
					class="regular-text"
					style="width: 250px;"
				/>
				<button type="button" id="rpm-clear-search" class="button" title="Clear search">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
			</div>
		</div>
		
		<div id="rpm-tags-stats" style="background: #f0f6fc; padding: 15px; border-left: 3px solid #2271b1; border-radius: 4px; margin-bottom: 20px;">
			<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Total Tags</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;" id="rpm-total-tags"><?php echo number_format_i18n( $tag_count ); ?></div>
				</div>
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Showing</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;" id="rpm-showing-tags"><?php echo number_format_i18n( $tag_count ); ?></div>
				</div>
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Most Used</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;">
						<?php
						$most_used = ! empty( $all_tags ) ? $all_tags[0]->name : 'N/A';
						$max_count = 0;
						foreach ( $all_tags as $tag ) {
							if ( $tag->count > $max_count ) {
								$max_count = $tag->count;
								$most_used = $tag->name;
							}
						}
						echo esc_html( $most_used );
						?>
					</div>
				</div>
			</div>
		</div>
		
		<div style="overflow-x: auto;">
			<table class="wp-list-table widefat fixed striped" id="rpm-tags-table">
				<thead>
					<tr>
						<th style="width: 40%;">Tag Name</th>
						<th style="width: 25%;">Slug</th>
						<th style="width: 15%; text-align: center;">Posts</th>
						<th style="width: 20%; text-align: center;">Actions</th>
					</tr>
				</thead>
				<tbody id="rpm-tags-tbody">
					<?php if ( ! empty( $all_tags ) ): ?>
						<?php foreach ( $all_tags as $tag ): ?>
							<tr class="rpm-tag-row" data-tag-name="<?php echo esc_attr( strtolower( $tag->name ) ); ?>">
								<td>
									<strong style="font-size: 14px;"><?php echo esc_html( $tag->name ); ?></strong>
								</td>
								<td>
									<code style="background: #f0f0f0; padding: 3px 8px; border-radius: 3px; font-size: 12px;">
										<?php echo esc_html( $tag->slug ); ?>
									</code>
								</td>
								<td style="text-align: center;">
									<span style="background: #2271b1; color: white; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600;">
										<?php echo number_format_i18n( $tag->count ); ?>
									</span>
								</td>
								<td style="text-align: center;">
									<a href="<?php echo admin_url( 'edit.php?tag=' . $tag->slug ); ?>" class="button button-small" title="View posts with this tag">
										<span class="dashicons dashicons-visibility" style="vertical-align: middle; margin-top: 4px;"></span>
										View Posts
									</a>
									<a href="<?php echo admin_url( 'term.php?taxonomy=post_tag&tag_ID=' . $tag->term_id ); ?>" class="button button-small" title="Edit tag">
										<span class="dashicons dashicons-edit" style="vertical-align: middle; margin-top: 4px;"></span>
										Edit
									</a>
									<button 
										type="button" 
										class="button button-small rpm-delete-tag" 
										data-tag-id="<?php echo esc_attr( $tag->term_id ); ?>"
										data-tag-name="<?php echo esc_attr( $tag->name ); ?>"
										title="Delete tag"
									>
										<span class="dashicons dashicons-trash" style="vertical-align: middle; margin-top: 4px; color: #d63638;"></span>
										Delete
									</button>
								</td>
							</tr>
						<?php endforeach; ?>
					<?php else: ?>
						<tr>
							<td colspan="4" style="text-align: center; padding: 40px;">
								<span class="dashicons dashicons-tag" style="font-size: 48px; color: #ddd;"></span>
								<p style="color: #666; margin-top: 10px;">No tags found. Add your first tag above!</p>
							</td>
						</tr>
					<?php endif; ?>
				</tbody>
			</table>
		</div>
		
		<div id="rpm-no-results" style="display: none; text-align: center; padding: 40px; background: #f9f9f9; border-radius: 4px; margin-top: 20px;">
			<span class="dashicons dashicons-search" style="font-size: 48px; color: #ddd;"></span>
			<p style="color: #666; margin-top: 10px; font-size: 16px;">No tags match your search.</p>
			<button type="button" id="rpm-clear-search-2" class="button button-primary" style="margin-top: 10px;">Clear Search</button>
		</div>
	</div>
	
	<!-- Hidden delete form -->
	<form method="post" id="rpm-delete-tag-form" style="display: none;">
		<?php wp_nonce_field( 'rpm_delete_tag_action', 'rpm_delete_tag_nonce' ); ?>
		<input type="hidden" name="tag_id" id="rpm-delete-tag-id" value="" />
		<input type="hidden" name="rpm_delete_tag" value="1" />
	</form>
</div>

<script>
jQuery(document).ready(function($) {
	// Handle bulk tag creation (comma-separated)
	$('#rpm-add-tag-form').on('submit', function(e) {
		var tagInput = $('#rpm_tag_name').val().trim();
		
		// Check if input contains commas (bulk add)
		if (tagInput.indexOf(',') !== -1) {
			e.preventDefault();
			
			var tags = tagInput.split(',').map(function(tag) {
				return tag.trim();
			}).filter(function(tag) {
				return tag.length > 0;
			});
			
			if (tags.length === 0) {
				return;
			}
			
			// Disable form
			$(this).find('button[type="submit"]').prop('disabled', true).text('Adding...');
			$('#rpm-bulk-add-result').html('<div class="notice notice-info"><p>⏳ Adding ' + tags.length + ' tags...</p></div>');
			
			// Add tags one by one via AJAX
			var added = 0;
			var errors = [];
			var promises = [];
			
			tags.forEach(function(tagName) {
				var promise = $.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'rpm_add_tag',
						tag_name: tagName,
						nonce: $('#rpm_tag_nonce').val()
					}
				});
				
				promises.push(promise);
			});
			
			Promise.all(promises).then(function(results) {
				results.forEach(function(response, index) {
					if (response.success) {
						added++;
					} else {
						errors.push(tags[index] + ': ' + response.data);
					}
				});
				
				var html = '';
				if (added > 0) {
					html += '<div class="notice notice-success"><p>✅ Successfully added ' + added + ' tag(s)!</p></div>';
				}
				if (errors.length > 0) {
					html += '<div class="notice notice-error"><p>❌ Errors:<br>' + errors.join('<br>') + '</p></div>';
				}
				
				$('#rpm-bulk-add-result').html(html);
				$('#rpm_tag_name').val('');
				$('#rpm-add-tag-form button[type="submit"]').prop('disabled', false).html('<span class="dashicons dashicons-plus-alt" style="vertical-align: middle; margin-top: 8px;"></span> Add Tag');
				
				// Reload page after 2 seconds
				setTimeout(function() {
					location.reload();
				}, 2000);
			});
		}
	});
	
	// Search functionality
	$('#rpm-tag-search').on('keyup', function() {
		var searchTerm = $(this).val().toLowerCase().trim();
		filterTags(searchTerm);
	});
	
	$('#rpm-clear-search, #rpm-clear-search-2').on('click', function() {
		$('#rpm-tag-search').val('');
		filterTags('');
	});
	
	function filterTags(searchTerm) {
		var visibleCount = 0;
		
		$('.rpm-tag-row').each(function() {
			var tagName = $(this).data('tag-name');
			if (searchTerm === '' || tagName.indexOf(searchTerm) !== -1) {
				$(this).show();
				visibleCount++;
			} else {
				$(this).hide();
			}
		});
		
		$('#rpm-showing-tags').text(visibleCount.toLocaleString());
		
		if (visibleCount === 0) {
			$('#rpm-tags-table').hide();
			$('#rpm-no-results').show();
		} else {
			$('#rpm-tags-table').show();
			$('#rpm-no-results').hide();
		}
	}
	
	// Delete tag handler
	$('.rpm-delete-tag').on('click', function() {
		var tagId = $(this).data('tag-id');
		var tagName = $(this).data('tag-name');
		
		if (confirm('Are you sure you want to delete the tag "' + tagName + '"? This action cannot be undone.')) {
			$('#rpm-delete-tag-id').val(tagId);
			$('#rpm-delete-tag-form').submit();
		}
	});
	
	// ========================================
	// AI TAG SUGGESTIONS
	// ========================================
	
	$('#rpm_generate_tag_suggestions').on('click', function() {
		var $button = $(this);
		var selectedPages = [];
		
		$('input[name="rpm_suggest_pages[]"]:checked').each(function() {
			selectedPages.push($(this).val());
		});
		
		if (selectedPages.length === 0) {
			$('#rpm_tag_suggestions_status').html('<div class="notice notice-error"><p>❌ Please select at least one page to analyze.</p></div>');
			return;
		}
		
		var count = parseInt($('#rpm_suggest_tags_count').val());
		if (count < 5 || count > 50) {
			$('#rpm_tag_suggestions_status').html('<div class="notice notice-error"><p>❌ Please enter a valid number of tags (5-50).</p></div>');
			return;
		}
		
		// Disable button and show processing
		$button.prop('disabled', true).html('🤖 Analyzing Pages...');
		$('#rpm_tag_suggestions_status').html('<div class="notice notice-info"><p>⏳ <strong>AI is analyzing ' + selectedPages.length + ' page(s)...</strong> This may take 20-30 seconds.</p></div>');
		$('#rpm_tag_suggestions_results').hide();
		
		// Make AJAX request
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'rpm_suggest_tags',
				_wpnonce: '<?php echo wp_create_nonce('rpm_suggest_tags'); ?>',
				page_ids: selectedPages,
				count: count
			},
			timeout: 60000,
			success: function(response) {
				if (response.success) {
					var tags = response.data.tags || [];
					
					if (tags.length === 0) {
						$('#rpm_tag_suggestions_status').html('<div class="notice notice-warning"><p>⚠️ No tags were suggested. Try selecting different pages or adjusting the count.</p></div>');
					} else {
						// Display suggestions
						var html = '';
						tags.forEach(function(tag) {
							html += '<div style="background: #f0f6fc; border: 1px solid #2271b1; border-radius: 4px; padding: 10px;">';
							html += '<label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">';
							html += '<input type="checkbox" class="rpm-suggested-tag" value="' + tag + '" checked />';
							html += '<span style="font-weight: 600;">' + tag + '</span>';
							html += '</label>';
							html += '</div>';
						});
						
						$('#rpm_tag_suggestions_list').html(html);
						$('#rpm_tag_suggestions_results').show();
						$('#rpm_tag_suggestions_status').html('<div class="notice notice-success"><p>✅ <strong>AI generated ' + tags.length + ' tag suggestions!</strong> Review and select which tags to create.</p></div>');
					}
				} else {
					$('#rpm_tag_suggestions_status').html('<div class="notice notice-error"><p>❌ Error: ' + (response.data || 'Unknown error') + '</p></div>');
				}
			},
			error: function(xhr, status, error) {
				$('#rpm_tag_suggestions_status').html('<div class="notice notice-error"><p>❌ Request failed: ' + error + '</p></div>');
			},
			complete: function() {
				$button.prop('disabled', false).html('<span class="dashicons dashicons-lightbulb" style="vertical-align: middle; margin-top: 8px;"></span> 🤖 Generate Tag Suggestions');
			}
		});
	});
	
	// Select/Deselect All Tags
	$('#rpm_select_all_tags').on('click', function() {
		$('.rpm-suggested-tag').prop('checked', true);
	});
	
	$('#rpm_deselect_all_tags').on('click', function() {
		$('.rpm-suggested-tag').prop('checked', false);
	});
	
	// Create Selected Tags
	$('#rpm_create_selected_tags').on('click', function() {
		var $button = $(this);
		var selectedTags = [];
		
		$('.rpm-suggested-tag:checked').each(function() {
			selectedTags.push($(this).val());
		});
		
		if (selectedTags.length === 0) {
			alert('Please select at least one tag to create.');
			return;
		}
		
		if (!confirm('Create ' + selectedTags.length + ' new tag(s)?')) {
			return;
		}
		
		$button.prop('disabled', true).text('Creating...');
		$('#rpm_tag_suggestions_status').html('<div class="notice notice-info"><p>⏳ Creating ' + selectedTags.length + ' tag(s)...</p></div>');
		
		// Create tags via bulk add mechanism
		var added = 0;
		var errors = [];
		var promises = [];
		
		selectedTags.forEach(function(tagName) {
			var promise = $.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'rpm_add_tag',
					tag_name: tagName,
					nonce: $('#rpm_tag_nonce').val()
				}
			});
			promises.push(promise);
		});
		
		Promise.all(promises).then(function(results) {
			results.forEach(function(response, index) {
				if (response.success) {
					added++;
				} else {
					errors.push(selectedTags[index] + ': ' + response.data);
				}
			});
			
			var html = '';
			if (added > 0) {
				html += '<div class="notice notice-success"><p>✅ Successfully created ' + added + ' tag(s)!</p></div>';
			}
			if (errors.length > 0) {
				html += '<div class="notice notice-error"><p>❌ Errors:<br>' + errors.join('<br>') + '</p></div>';
			}
			
			$('#rpm_tag_suggestions_status').html(html);
			$('#rpm_tag_suggestions_results').hide();
			$button.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-top: 4px;"></span> Create Selected Tags');
			
			// Reload page after 2 seconds
			setTimeout(function() {
				location.reload();
			}, 2000);
		});
	});
});
</script>

<style>
.rpm-tag-row:hover {
	background-color: #f0f6fc !important;
}

.button-small {
	padding: 4px 8px;
	height: auto;
	line-height: 1.4;
	font-size: 12px;
}

#rpm-tags-table th {
	background: #f0f6fc;
	font-weight: 600;
	padding: 12px;
}

#rpm-tags-table td {
	padding: 12px;
}
</style>
