<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// Handle category actions
$message = '';
$message_type = '';

if ( isset( $_POST['rpm_add_category'] ) && check_admin_referer( 'rpm_add_category_action', 'rpm_category_nonce' ) ) {
	$category_name = sanitize_text_field( $_POST['category_name'] );
	$parent_id = isset( $_POST['parent_category'] ) ? intval( $_POST['parent_category'] ) : 0;
	
	if ( ! empty( $category_name ) ) {
		$result = wp_insert_term( $category_name, 'category', array(
			'parent' => $parent_id
		));
		
		if ( ! is_wp_error( $result ) ) {
			$message = 'Category "' . esc_html( $category_name ) . '" added successfully!';
			$message_type = 'updated';
		} else {
			$message = 'Error: ' . $result->get_error_message();
			$message_type = 'error';
		}
	} else {
		$message = 'Please enter a category name.';
		$message_type = 'error';
	}
}

if ( isset( $_POST['rpm_delete_category'] ) && check_admin_referer( 'rpm_delete_category_action', 'rpm_delete_category_nonce' ) ) {
	$category_id = intval( $_POST['category_id'] );
	
	// Prevent deletion of default category (usually ID 1)
	$default_cat = get_option( 'default_category' );
	if ( $category_id == $default_cat ) {
		$message = 'Cannot delete the default category.';
		$message_type = 'error';
	} else {
		$result = wp_delete_term( $category_id, 'category' );
		if ( ! is_wp_error( $result ) && $result ) {
			$message = 'Category deleted successfully!';
			$message_type = 'updated';
		} else {
			$message = 'Error deleting category.';
			$message_type = 'error';
		}
	}
}

// Get all categories
$args = array(
	'taxonomy' => 'category',
	'hide_empty' => false,
	'orderby' => 'name',
	'order' => 'ASC'
);
$all_categories = get_terms( $args );
$category_count = count( $all_categories );
$default_cat_id = get_option( 'default_category' );
?>

<div class="wrap">
	<h1 class="wp-heading-inline">
		<span class="dashicons dashicons-category" style="font-size: 28px; width: 28px; height: 28px; vertical-align: middle;"></span>
		Categories Manager
	</h1>
	<p class="description" style="margin-top: 10px; font-size: 14px;">Manage WordPress categories used by AI-generated posts. View, edit, and organize categories for your content.</p>
	<hr class="wp-header-end">
	
	<?php if ( ! empty( $message ) ): ?>
		<div class="notice notice-<?php echo esc_attr( $message_type ); ?> is-dismissible">
			<p><?php echo esc_html( $message ); ?></p>
		</div>
	<?php endif; ?>
	
	<div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<h2 style="margin-top: 0;">
			<span class="dashicons dashicons-plus-alt" style="font-size: 20px; width: 20px; height: 20px; vertical-align: middle;"></span>
			Add New Category
		</h2>
		<p class="description">Create a new category for organizing your posts. You can also assign a parent category to create a hierarchy.</p>
		
		<form method="post" action="" id="rpm-add-category-form" style="margin-top: 20px;">
			<?php wp_nonce_field( 'rpm_add_category_action', 'rpm_category_nonce' ); ?>
			
			<div style="display: flex; gap: 10px; align-items: flex-start; flex-wrap: wrap;">
				<div style="flex: 1; min-width: 300px;">
					<input 
						type="text" 
						name="category_name" 
						id="rpm_category_name" 
						class="regular-text" 
						placeholder="Enter category name (e.g., Technology, Lifestyle)"
						style="width: 100%; font-size: 14px; padding: 8px;"
					/>
					<p class="description" style="margin-top: 5px;">
						💡 Tip: Separate multiple categories with commas for bulk creation
					</p>
				</div>
				
				<div style="min-width: 200px;">
					<select name="parent_category" id="rpm_parent_category" class="regular-text" style="width: 100%; font-size: 14px; padding: 8px;">
						<option value="0">No Parent (Top Level)</option>
						<?php foreach ( $all_categories as $cat ): ?>
							<option value="<?php echo esc_attr( $cat->term_id ); ?>">
								<?php echo esc_html( $cat->name ); ?>
								<?php if ( $cat->parent ): ?>
									(Sub-category)
								<?php endif; ?>
							</option>
						<?php endforeach; ?>
					</select>
					<p class="description" style="margin-top: 5px;">Parent Category (optional)</p>
				</div>
				
				<button type="submit" name="rpm_add_category" class="button button-primary" style="height: 40px; padding: 0 20px;">
					<span class="dashicons dashicons-plus-alt" style="vertical-align: middle; margin-top: 8px;"></span>
					Add Category
				</button>
			</div>
		</form>
		
		<div id="rpm-bulk-add-result" style="margin-top: 15px;"></div>
	</div>
	
	<!-- AI-Powered Category Suggestions -->
	<div class="ai-powered-suggestions" style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<h2 style="margin-top: 0;">
			<span class="dashicons dashicons-lightbulb" style="font-size: 20px; color: #f0b849; vertical-align: middle;"></span>
			🤖 AI-Powered Category Suggestions
		</h2>
		<p class="description">
			Let AI analyze your site's content and suggest relevant categories. Select pages to analyze and specify how many categories you'd like.
		</p>
		
		<!-- Info Banner -->
		<div style="background: #f0f6fc; border-left: 3px solid #2271b1; padding: 15px; margin: 20px 0; border-radius: 4px;">
			<strong>💡 How it works:</strong>
			<ol style="margin: 10px 0 0 20px; padding: 0;">
				<li>Select pages that represent your site's core content</li>
				<li>Choose how many categories you want (3-20 categories)</li>
				<li>Click "Generate Suggestions" and wait 20-30 seconds</li>
				<li>Review AI suggestions and select which categories to create</li>
			</ol>
		</div>
		
		<!-- Page Selector -->
		<div style="margin: 20px 0;">
			<h3 style="margin-bottom: 10px;">📄 Select Pages to Analyze:</h3>
			<div style="max-height: 300px; overflow-y: auto; border: 1px solid #c3c4c7; border-radius: 4px; padding: 15px; background: #f9f9f9;">
				<?php
				$pages = get_pages(['post_status' => 'publish', 'number' => 100, 'sort_column' => 'post_modified']);
				$homepage_id = get_option('page_on_front');
				
				if (empty($pages)) {
					echo '<p style="color: #666; font-style: italic;">No published pages found.</p>';
				} else {
					foreach ($pages as $page) {
						$is_homepage = ($page->ID == $homepage_id);
						$checked = $is_homepage ? 'checked' : '';
						echo '<div style="margin-bottom: 8px;">';
						echo '<label style="display: flex; align-items: center; gap: 8px;">';
						echo '<input type="checkbox" name="rpm_suggest_category_pages[]" value="' . esc_attr($page->ID) . '" ' . $checked . ' />';
						echo '<span>' . esc_html($page->post_title);
						if ($is_homepage) echo ' <span style="background: #00a32a; color: white; padding: 2px 6px; border-radius: 3px; font-size: 10px; margin-left: 5px;">HOME</span>';
						echo '</span>';
						echo '</label>';
						echo '</div>';
					}
				}
				?>
			</div>
		</div>
		
		<!-- Count Input & Generate Button -->
		<div style="display: flex; gap: 15px; align-items: flex-end; margin-top: 20px;">
			<div style="flex: 1;">
				<label for="rpm_suggest_categories_count" style="display: block; margin-bottom: 5px; font-weight: 600;">
					🔢 How many categories to suggest?
				</label>
				<input 
					type="number" 
					id="rpm_suggest_categories_count" 
					name="rpm_suggest_categories_count" 
					value="5" 
					min="3" 
					max="20" 
					class="regular-text"
					style="width: 150px;"
				/>
				<p class="description" style="margin-top: 5px;">Between 3 and 20 categories</p>
			</div>
			
			<button type="button" id="rpm_generate_category_suggestions" class="button button-primary" style="height: 40px; padding: 0 20px;">
				<span class="dashicons dashicons-lightbulb" style="vertical-align: middle; margin-top: 8px;"></span>
				🤖 Generate Category Suggestions
			</button>
		</div>
		
		<!-- Status Area -->
		<div id="rpm_category_suggestions_status" style="margin-top: 20px;"></div>
		
		<!-- Results Area (initially hidden) -->
		<div id="rpm_category_suggestions_results" style="display: none; margin-top: 20px;">
			<h3 style="margin-bottom: 15px;">✨ AI Suggested Categories:</h3>
			<p class="description" style="margin-bottom: 15px;">
				Select the categories you want to create. Optionally, choose a parent category for all selected categories.
			</p>
			
			<!-- Parent Category Selector -->
			<div style="margin-bottom: 15px;">
				<label for="rpm_suggested_categories_parent" style="display: block; margin-bottom: 5px; font-weight: 600;">
					📂 Parent Category (optional):
				</label>
				<select id="rpm_suggested_categories_parent" class="regular-text" style="width: 250px;">
					<option value="0">No Parent (Top Level)</option>
					<?php foreach ( $all_categories as $cat ): ?>
						<option value="<?php echo esc_attr( $cat->term_id ); ?>">
							<?php echo esc_html( $cat->name ); ?>
						</option>
					<?php endforeach; ?>
				</select>
			</div>
			
			<div id="rpm_category_suggestions_list" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px; margin-bottom: 20px;">
				<!-- Categories will be dynamically inserted here -->
			</div>
			
			<div style="display: flex; gap: 10px;">
				<button type="button" id="rpm_create_selected_categories" class="button button-primary">
					<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-top: 4px;"></span>
					Create Selected Categories
				</button>
				<button type="button" id="rpm_select_all_categories" class="button">Select All</button>
				<button type="button" id="rpm_deselect_all_categories" class="button">Deselect All</button>
			</div>
		</div>
	</div>
	
	<div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-top: 20px;">
		<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
			<h2 style="margin: 0;">
				<span class="dashicons dashicons-list-view" style="font-size: 20px; width: 20px; height: 20px; vertical-align: middle;"></span>
				All Categories (<?php echo number_format_i18n( $category_count ); ?>)
			</h2>
			
			<div style="display: flex; gap: 10px; align-items: center;">
				<input 
					type="text" 
					id="rpm-category-search" 
					placeholder="🔍 Search categories..." 
					class="regular-text"
					style="width: 250px;"
				/>
				<button type="button" id="rpm-clear-search" class="button" title="Clear search">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
			</div>
		</div>
		
		<div id="rpm-categories-stats" style="background: #f0f6fc; padding: 15px; border-left: 3px solid #2271b1; border-radius: 4px; margin-bottom: 20px;">
			<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Total Categories</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;" id="rpm-total-categories"><?php echo number_format_i18n( $category_count ); ?></div>
				</div>
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Showing</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;" id="rpm-showing-categories"><?php echo number_format_i18n( $category_count ); ?></div>
				</div>
				<div>
					<div style="color: #666; font-size: 12px; text-transform: uppercase; margin-bottom: 5px;">Most Used</div>
					<div style="font-size: 24px; font-weight: bold; color: #2271b1;">
						<?php
						$most_used = ! empty( $all_categories ) ? $all_categories[0]->name : 'N/A';
						$max_count = 0;
						foreach ( $all_categories as $cat ) {
							if ( $cat->count > $max_count ) {
								$max_count = $cat->count;
								$most_used = $cat->name;
							}
						}
						echo esc_html( $most_used );
						?>
					</div>
				</div>
			</div>
		</div>
		
		<div style="overflow-x: auto;">
			<table class="wp-list-table widefat fixed striped" id="rpm-categories-table">
				<thead>
					<tr>
						<th style="width: 35%;">Category Name</th>
						<th style="width: 20%;">Slug</th>
						<th style="width: 15%;">Parent</th>
						<th style="width: 10%; text-align: center;">Posts</th>
						<th style="width: 20%; text-align: center;">Actions</th>
					</tr>
				</thead>
				<tbody id="rpm-categories-tbody">
					<?php if ( ! empty( $all_categories ) ): ?>
						<?php foreach ( $all_categories as $category ): ?>
							<?php
							$parent_name = '';
							if ( $category->parent ) {
								$parent = get_term( $category->parent, 'category' );
								if ( $parent && ! is_wp_error( $parent ) ) {
									$parent_name = $parent->name;
								}
							}
							$is_default = ( $category->term_id == $default_cat_id );
							?>
							<tr class="rpm-category-row" data-category-name="<?php echo esc_attr( strtolower( $category->name ) ); ?>">
								<td>
									<strong style="font-size: 14px;">
										<?php echo esc_html( $category->name ); ?>
										<?php if ( $is_default ): ?>
											<span style="background: #00a32a; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600; margin-left: 8px;">DEFAULT</span>
										<?php endif; ?>
									</strong>
								</td>
								<td>
									<code style="background: #f0f0f0; padding: 3px 8px; border-radius: 3px; font-size: 12px;">
										<?php echo esc_html( $category->slug ); ?>
									</code>
								</td>
								<td>
									<?php if ( $parent_name ): ?>
										<span style="color: #666; font-size: 13px;">
											↳ <?php echo esc_html( $parent_name ); ?>
										</span>
									<?php else: ?>
										<span style="color: #999; font-size: 13px; font-style: italic;">—</span>
									<?php endif; ?>
								</td>
								<td style="text-align: center;">
									<span style="background: #2271b1; color: white; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600;">
										<?php echo number_format_i18n( $category->count ); ?>
									</span>
								</td>
								<td style="text-align: center;">
									<a href="<?php echo admin_url( 'edit.php?category_name=' . $category->slug ); ?>" class="button button-small" title="View posts in this category">
										<span class="dashicons dashicons-visibility" style="vertical-align: middle; margin-top: 4px;"></span>
										View Posts
									</a>
									<a href="<?php echo admin_url( 'term.php?taxonomy=category&tag_ID=' . $category->term_id ); ?>" class="button button-small" title="Edit category">
										<span class="dashicons dashicons-edit" style="vertical-align: middle; margin-top: 4px;"></span>
										Edit
									</a>
									<button 
										type="button" 
										class="button button-small rpm-delete-category" 
										data-category-id="<?php echo esc_attr( $category->term_id ); ?>"
										data-category-name="<?php echo esc_attr( $category->name ); ?>"
										title="<?php echo $is_default ? 'Cannot delete default category' : 'Delete category'; ?>"
										<?php echo $is_default ? 'disabled' : ''; ?>
										style="<?php echo $is_default ? 'opacity: 0.5; cursor: not-allowed;' : ''; ?>"
									>
										<span class="dashicons dashicons-trash" style="vertical-align: middle; margin-top: 4px; color: #d63638;"></span>
										Delete
									</button>
								</td>
							</tr>
						<?php endforeach; ?>
					<?php else: ?>
						<tr>
							<td colspan="5" style="text-align: center; padding: 40px;">
								<span class="dashicons dashicons-category" style="font-size: 48px; color: #ddd;"></span>
								<p style="color: #666; margin-top: 10px;">No categories found. Add your first category above!</p>
							</td>
						</tr>
					<?php endif; ?>
				</tbody>
			</table>
		</div>
		
		<div id="rpm-no-results" style="display: none; text-align: center; padding: 40px; background: #f9f9f9; border-radius: 4px; margin-top: 20px;">
			<span class="dashicons dashicons-search" style="font-size: 48px; color: #ddd;"></span>
			<p style="color: #666; margin-top: 10px; font-size: 16px;">No categories match your search.</p>
			<button type="button" id="rpm-clear-search-2" class="button button-primary" style="margin-top: 10px;">Clear Search</button>
		</div>
	</div>
	
	<!-- Hidden delete form -->
	<form method="post" id="rpm-delete-category-form" style="display: none;">
		<?php wp_nonce_field( 'rpm_delete_category_action', 'rpm_delete_category_nonce' ); ?>
		<input type="hidden" name="category_id" id="rpm-delete-category-id" value="" />
		<input type="hidden" name="rpm_delete_category" value="1" />
	</form>
</div>

<script>
jQuery(document).ready(function($) {
	// Handle bulk category creation (comma-separated)
	$('#rpm-add-category-form').on('submit', function(e) {
		var categoryInput = $('#rpm_category_name').val().trim();
		
		// Check if input contains commas (bulk add)
		if (categoryInput.indexOf(',') !== -1) {
			e.preventDefault();
			
			var categories = categoryInput.split(',').map(function(cat) {
				return cat.trim();
			}).filter(function(cat) {
				return cat.length > 0;
			});
			
			if (categories.length === 0) {
				return;
			}
			
			// Disable form
			$(this).find('button[type="submit"]').prop('disabled', true).text('Adding...');
			$('#rpm-bulk-add-result').html('<div class="notice notice-info"><p>⏳ Adding ' + categories.length + ' categories...</p></div>');
			
			var parentId = $('#rpm_parent_category').val();
			
			// Add categories one by one via AJAX
			var added = 0;
			var errors = [];
			var promises = [];
			
			categories.forEach(function(categoryName) {
				var promise = $.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'rpm_add_category',
						category_name: categoryName,
						parent_id: parentId,
						nonce: $('#rpm_category_nonce').val()
					}
				});
				
				promises.push(promise);
			});
			
			Promise.all(promises).then(function(results) {
				results.forEach(function(response, index) {
					if (response.success) {
						added++;
					} else {
						errors.push(categories[index] + ': ' + response.data);
					}
				});
				
				var html = '';
				if (added > 0) {
					html += '<div class="notice notice-success"><p>✅ Successfully added ' + added + ' category(ies)!</p></div>';
				}
				if (errors.length > 0) {
					html += '<div class="notice notice-error"><p>❌ Errors:<br>' + errors.join('<br>') + '</p></div>';
				}
				
				$('#rpm-bulk-add-result').html(html);
				$('#rpm_category_name').val('');
				$('#rpm-add-category-form button[type="submit"]').prop('disabled', false).html('<span class="dashicons dashicons-plus-alt" style="vertical-align: middle; margin-top: 8px;"></span> Add Category');
				
				// Reload page after 2 seconds
				setTimeout(function() {
					location.reload();
				}, 2000);
			});
		}
	});
	
	// Search functionality
	$('#rpm-category-search').on('keyup', function() {
		var searchTerm = $(this).val().toLowerCase().trim();
		filterCategories(searchTerm);
	});
	
	$('#rpm-clear-search, #rpm-clear-search-2').on('click', function() {
		$('#rpm-category-search').val('');
		filterCategories('');
	});
	
	function filterCategories(searchTerm) {
		var visibleCount = 0;
		
		$('.rpm-category-row').each(function() {
			var categoryName = $(this).data('category-name');
			if (searchTerm === '' || categoryName.indexOf(searchTerm) !== -1) {
				$(this).show();
				visibleCount++;
			} else {
				$(this).hide();
			}
		});
		
		$('#rpm-showing-categories').text(visibleCount.toLocaleString());
		
		if (visibleCount === 0) {
			$('#rpm-categories-table').hide();
			$('#rpm-no-results').show();
		} else {
			$('#rpm-categories-table').show();
			$('#rpm-no-results').hide();
		}
	}
	
	// Delete category handler
	$('.rpm-delete-category').on('click', function() {
		var categoryId = $(this).data('category-id');
		var categoryName = $(this).data('category-name');
		
		if (confirm('Are you sure you want to delete the category "' + categoryName + '"? Posts in this category will be moved to the default category.')) {
			$('#rpm-delete-category-id').val(categoryId);
			$('#rpm-delete-category-form').submit();
		}
	});
	
	// ========================================
	// AI CATEGORY SUGGESTIONS
	// ========================================
	
	$('#rpm_generate_category_suggestions').on('click', function() {
		var $button = $(this);
		var selectedPages = [];
		
		$('input[name="rpm_suggest_category_pages[]"]:checked').each(function() {
			selectedPages.push($(this).val());
		});
		
		if (selectedPages.length === 0) {
			$('#rpm_category_suggestions_status').html('<div class="notice notice-error"><p>❌ Please select at least one page to analyze.</p></div>');
			return;
		}
		
		var count = parseInt($('#rpm_suggest_categories_count').val());
		if (count < 3 || count > 20) {
			$('#rpm_category_suggestions_status').html('<div class="notice notice-error"><p>❌ Please enter a valid number of categories (3-20).</p></div>');
			return;
		}
		
		// Disable button and show processing
		$button.prop('disabled', true).html('🤖 Analyzing Pages...');
		$('#rpm_category_suggestions_status').html('<div class="notice notice-info"><p>⏳ <strong>AI is analyzing ' + selectedPages.length + ' page(s)...</strong> This may take 20-30 seconds.</p></div>');
		$('#rpm_category_suggestions_results').hide();
		
		// Make AJAX request
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'rpm_suggest_categories',
				_wpnonce: '<?php echo wp_create_nonce('rpm_suggest_categories'); ?>',
				page_ids: selectedPages,
				count: count
			},
			timeout: 60000,
			success: function(response) {
				if (response.success) {
					var categories = response.data.categories || [];
					
					if (categories.length === 0) {
						$('#rpm_category_suggestions_status').html('<div class="notice notice-warning"><p>⚠️ No categories were suggested. Try selecting different pages or adjusting the count.</p></div>');
					} else {
						// Display suggestions
						var html = '';
						categories.forEach(function(category) {
							html += '<div style="background: #f0f6fc; border: 1px solid #2271b1; border-radius: 4px; padding: 10px;">';
							html += '<label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">';
							html += '<input type="checkbox" class="rpm-suggested-category" value="' + category + '" checked />';
							html += '<span style="font-weight: 600;">' + category + '</span>';
							html += '</label>';
							html += '</div>';
						});
						
						$('#rpm_category_suggestions_list').html(html);
						$('#rpm_category_suggestions_results').show();
						$('#rpm_category_suggestions_status').html('<div class="notice notice-success"><p>✅ <strong>AI generated ' + categories.length + ' category suggestions!</strong> Review and select which categories to create.</p></div>');
					}
				} else {
					$('#rpm_category_suggestions_status').html('<div class="notice notice-error"><p>❌ Error: ' + (response.data || 'Unknown error') + '</p></div>');
				}
			},
			error: function(xhr, status, error) {
				$('#rpm_category_suggestions_status').html('<div class="notice notice-error"><p>❌ Request failed: ' + error + '</p></div>');
			},
			complete: function() {
				$button.prop('disabled', false).html('<span class="dashicons dashicons-lightbulb" style="vertical-align: middle; margin-top: 8px;"></span> 🤖 Generate Category Suggestions');
			}
		});
	});
	
	// Select/Deselect All Categories
	$('#rpm_select_all_categories').on('click', function() {
		$('.rpm-suggested-category').prop('checked', true);
	});
	
	$('#rpm_deselect_all_categories').on('click', function() {
		$('.rpm-suggested-category').prop('checked', false);
	});
	
	// Create Selected Categories
	$('#rpm_create_selected_categories').on('click', function() {
		var $button = $(this);
		var selectedCategories = [];
		
		$('.rpm-suggested-category:checked').each(function() {
			selectedCategories.push($(this).val());
		});
		
		if (selectedCategories.length === 0) {
			alert('Please select at least one category to create.');
			return;
		}
		
		if (!confirm('Create ' + selectedCategories.length + ' new category(ies)?')) {
			return;
		}
		
		var parentId = $('#rpm_suggested_categories_parent').val();
		
		$button.prop('disabled', true).text('Creating...');
		$('#rpm_category_suggestions_status').html('<div class="notice notice-info"><p>⏳ Creating ' + selectedCategories.length + ' category(ies)...</p></div>');
		
		// Create categories via bulk add mechanism
		var added = 0;
		var errors = [];
		var promises = [];
		
		selectedCategories.forEach(function(categoryName) {
			var promise = $.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'rpm_add_category',
					category_name: categoryName,
					parent_id: parentId,
					nonce: $('#rpm_category_nonce').val()
				}
			});
			promises.push(promise);
		});
		
		Promise.all(promises).then(function(results) {
			results.forEach(function(response, index) {
				if (response.success) {
					added++;
				} else {
					errors.push(selectedCategories[index] + ': ' + response.data);
				}
			});
			
			var html = '';
			if (added > 0) {
				html += '<div class="notice notice-success"><p>✅ Successfully created ' + added + ' category(ies)!</p></div>';
			}
			if (errors.length > 0) {
				html += '<div class="notice notice-error"><p>❌ Errors:<br>' + errors.join('<br>') + '</p></div>';
			}
			
			$('#rpm_category_suggestions_status').html(html);
			$('#rpm_category_suggestions_results').hide();
			$button.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-top: 4px;"></span> Create Selected Categories');
			
			// Reload page after 2 seconds
			setTimeout(function() {
				location.reload();
			}, 2000);
		});
	});
});
</script>

<style>
.rpm-category-row:hover {
	background-color: #f0f6fc !important;
}

.button-small {
	padding: 4px 8px;
	height: auto;
	line-height: 1.4;
	font-size: 12px;
}

#rpm-categories-table th {
	background: #f0f6fc;
	font-weight: 600;
	padding: 12px;
}

#rpm-categories-table td {
	padding: 12px;
}
</style>
