<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// Load WP_List_Table if not loaded
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

// Handle empty trash action
if ( isset( $_POST['rpm_empty_trash'] ) && check_admin_referer( 'rpm_empty_trash' ) ) {
	$trashed_posts = get_posts( array(
		'post_type'      => 'post',
		'post_status'    => 'trash',
		'posts_per_page' => -1,
		'fields'         => 'ids',
		'meta_query'     => array(
			array(
				'key'     => '_rpm_auto_posted',
				'value'   => '1',
				'compare' => '='
			)
		)
	));
	
	$deleted_count = 0;
	foreach ( $trashed_posts as $post_id ) {
		if ( wp_delete_post( $post_id, true ) ) {
			$deleted_count++;
		}
	}
	
	if ( $deleted_count > 0 ) {
		echo '<div class="notice notice-success is-dismissible"><p>';
		printf( esc_html__( '🗑️ Emptied trash: %d post(s) permanently deleted.', 'remote-post-manager' ), $deleted_count );
		echo '</p></div>';
	} else {
		echo '<div class="notice notice-info is-dismissible"><p>';
		esc_html_e( 'Trash is already empty.', 'remote-post-manager' );
		echo '</p></div>';
	}
}

// Handle bulk actions
if ( isset( $_POST['action'] ) || isset( $_POST['action2'] ) ) {
	$action = '';
	if ( ! empty( $_POST['action'] ) && $_POST['action'] !== '-1' ) {
		$action = $_POST['action'];
	} elseif ( ! empty( $_POST['action2'] ) && $_POST['action2'] !== '-1' ) {
		$action = $_POST['action2'];
	}
	
	if ( $action && ! empty( $_POST['post'] ) && is_array( $_POST['post'] ) ) {
		check_admin_referer( 'bulk-auto_posts' );
		
		$post_ids = array_map( 'intval', $_POST['post'] );
		$count = 0;
		
		foreach ( $post_ids as $post_id ) {
			switch ( $action ) {
				case 'publish':
					if ( wp_update_post( array( 'ID' => $post_id, 'post_status' => 'publish' ) ) ) {
						$count++;
					}
					break;
				case 'draft':
					if ( wp_update_post( array( 'ID' => $post_id, 'post_status' => 'draft' ) ) ) {
						$count++;
					}
					break;
				case 'pending':
					if ( wp_update_post( array( 'ID' => $post_id, 'post_status' => 'pending' ) ) ) {
						$count++;
					}
					break;
				case 'trash':
					if ( wp_trash_post( $post_id ) ) {
						$count++;
					}
					break;
				case 'delete':
					if ( wp_delete_post( $post_id, true ) ) {
						$count++;
					}
					break;
			}
		}
		
		if ( $count > 0 ) {
			$action_labels = array(
				'publish' => 'published',
				'draft'   => 'set to draft',
				'pending' => 'set to pending',
				'trash'   => 'moved to trash',
				'delete'  => 'permanently deleted'
			);
			$action_label = isset( $action_labels[ $action ] ) ? $action_labels[ $action ] : 'updated';
			
			echo '<div class="notice notice-success is-dismissible"><p>';
			printf( esc_html__( '%d post(s) %s.', 'remote-post-manager' ), $count, $action_label );
			echo '</p></div>';
		}
	}
}

class RPM_Auto_Posts_Table extends WP_List_Table {
	
	public function __construct() {
		parent::__construct( array(
			'singular' => 'auto_post',
			'plural'   => 'auto_posts',
			'ajax'     => false
		));
	}
	
	public function get_columns() {
		return array(
			'cb'         => '<input type="checkbox" />',
			'title'      => __( 'Title', 'remote-post-manager' ),
			'status'     => __( 'Status', 'remote-post-manager' ),
			'categories' => __( 'Categories', 'remote-post-manager' ),
			'tags'       => __( 'Tags', 'remote-post-manager' ),
			'date'       => __( 'Date', 'remote-post-manager' )
		);
	}
	
	public function get_sortable_columns() {
		return array(
			'title' => array( 'title', false ),
			'date'  => array( 'date', true )
		);
	}
	
	public function get_bulk_actions() {
		return array(
			'publish' => __( 'Publish', 'remote-post-manager' ),
			'draft'   => __( 'Set to Draft', 'remote-post-manager' ),
			'pending' => __( 'Set to Pending', 'remote-post-manager' ),
			'trash'   => __( 'Move to Trash', 'remote-post-manager' ),
			'delete'  => __( 'Delete Permanently', 'remote-post-manager' )
		);
	}
	
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'status':
			case 'categories':
			case 'tags':
			case 'date':
				return $item[ $column_name ];
			default:
				return print_r( $item, true );
		}
	}
	
	public function column_cb( $item ) {
		return sprintf( '<input type="checkbox" name="post[]" value="%s" />', $item['ID'] );
	}
	
	public function column_title( $item ) {
		$edit_url = get_edit_post_link( $item['ID'] );
		$view_url = get_permalink( $item['ID'] );
		$delete_url = get_delete_post_link( $item['ID'] );
		
		$actions = array(
			'edit'   => sprintf( '<a href="%s">Edit</a>', esc_url( $edit_url ) ),
			'view'   => sprintf( '<a href="%s" target="_blank">View</a>', esc_url( $view_url ) ),
		);
		
		// Add "Change Image" action - opens AI image picker modal
		$actions['change_image'] = sprintf( '<a href="#" class="rpm-change-image" data-post-id="%d">Change Image</a>', $item['ID'] );
		
		$actions['delete'] = sprintf( '<a href="%s" onclick="return confirm(\'Delete this post?\');">Delete</a>', esc_url( $delete_url ) );
		
		return sprintf( '<strong><a href="%s">%s</a></strong>%s',
			esc_url( $edit_url ),
			esc_html( $item['title'] ),
			$this->row_actions( $actions )
		);
	}
	
	public function prepare_items() {
		$per_page = 20;
		$current_page = $this->get_pagenum();
		$orderby = isset( $_REQUEST['orderby'] ) ? sanitize_text_field( $_REQUEST['orderby'] ) : 'date';
		$order = isset( $_REQUEST['order'] ) ? sanitize_text_field( $_REQUEST['order'] ) : 'DESC';
		
		// Query for auto-generated posts
		$args = array(
			'post_type'      => 'post',
			'posts_per_page' => $per_page,
			'paged'          => $current_page,
			'orderby'        => $orderby,
			'order'          => $order,
			'meta_query'     => array(
				array(
					'key'     => '_rpm_auto_posted',
					'value'   => '1',
					'compare' => '='
				)
			)
		);
		
		$query = new WP_Query( $args );
		$posts = $query->posts;
		$total_items = $query->found_posts;
		
		// Format data for table
		$data = array();
		foreach ( $posts as $post ) {
			
			// Get categories
			$categories = wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) );
			$categories_display = ! empty( $categories ) ? implode( ', ', $categories ) : '—';
			
			// Get tags
			$tags = wp_get_post_tags( $post->ID, array( 'fields' => 'names' ) );
			$tags_display = ! empty( $tags ) ? implode( ', ', array_slice( $tags, 0, 3 ) ) : '—';
			if ( count( $tags ) > 3 ) {
				$tags_display .= ' +' . ( count( $tags ) - 3 );
			}
			
			// Status badge
			$status_colors = array(
				'publish' => '#00a32a',
				'draft'   => '#dba617',
				'pending' => '#d63638',
				'private' => '#8c8f94'
			);
			$status_color = isset( $status_colors[ $post->post_status ] ) ? $status_colors[ $post->post_status ] : '#8c8f94';
			$status_display = sprintf(
				'<span style="background: %s; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;">%s</span>',
				$status_color,
				ucfirst( $post->post_status )
			);
			
			$data[] = array(
				'ID'         => $post->ID,
				'title'      => $post->post_title,
				'status'     => $status_display,
				'categories' => $categories_display,
				'tags'       => $tags_display,
				'date'       => get_the_date( 'M j, Y g:i a', $post )
			);
		}
		
		$this->items = $data;
		
		$this->set_pagination_args( array(
			'total_items' => $total_items,
			'per_page'    => $per_page,
			'total_pages' => ceil( $total_items / $per_page )
		));
		
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();
		
		$this->_column_headers = array( $columns, $hidden, $sortable );
	}
}

// Render the page
?>
<div class="wrap">
	<h1 class="wp-heading-inline">
		<span class="dashicons dashicons-edit-page" style="font-size: 28px; width: 28px; height: 28px; vertical-align: middle;"></span>
		<?php esc_html_e( 'Auto-Generated Posts', 'remote-post-manager' ); ?>
	</h1>
	<p class="description" style="margin-top: 10px; font-size: 14px;">View and manage AI-generated posts. Edit, publish, or delete posts created by the auto-posting system.</p>
	
	<hr class="wp-header-end">
	
	<?php
	// Get stats
	$total_auto = new WP_Query( array(
		'post_type'      => 'post',
		'posts_per_page' => 1,
		'fields'         => 'ids',
		'meta_query'     => array(
			array(
				'key'     => '_rpm_auto_posted',
				'value'   => '1',
				'compare' => '='
			)
		)
	));
	
	$published = new WP_Query( array(
		'post_type'      => 'post',
		'post_status'    => 'publish',
		'posts_per_page' => 1,
		'fields'         => 'ids',
		'meta_query'     => array(
			array(
				'key'     => '_rpm_auto_posted',
				'value'   => '1',
				'compare' => '='
			)
		)
	));
	
	$pending = new WP_Query( array(
		'post_type'      => 'post',
		'post_status'    => 'pending',
		'posts_per_page' => 1,
		'fields'         => 'ids',
		'meta_query'     => array(
			array(
				'key'     => '_rpm_auto_posted',
				'value'   => '1',
				'compare' => '='
			)
		)
	));
	?>
	
	<div class="rpm-stats-boxes">
		<div class="rpm-stat-box">
			<div class="rpm-stat-number"><?php echo number_format_i18n( $total_auto->found_posts ); ?></div>
			<div class="rpm-stat-label">Total Posts</div>
		</div>
		<div class="rpm-stat-box">
			<div class="rpm-stat-number" style="color: #00a32a;"><?php echo number_format_i18n( $published->found_posts ); ?></div>
			<div class="rpm-stat-label">Published</div>
		</div>
		<div class="rpm-stat-box">
			<div class="rpm-stat-number" style="color: #d63638;"><?php echo number_format_i18n( $pending->found_posts ); ?></div>
			<div class="rpm-stat-label">Pending Review</div>
		</div>
		<div class="rpm-stat-box">
			<div class="rpm-stat-number" style="color: #2271b1;"><?php 
				$topics = get_option( 'rpm_topic_queue', array() );
				echo number_format_i18n( count( $topics ) ); 
			?></div>
			<div class="rpm-stat-label">Topics in Queue</div>
		</div>
	</div>
	
	<form method="post">
		<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>" />
		<?php
		$table = new RPM_Auto_Posts_Table();
		$table->prepare_items();
		$table->display();
		?>
	</form>
	
	<?php
	// Check if there are trashed posts
	$trashed_count = new WP_Query( array(
		'post_type'      => 'post',
		'post_status'    => 'trash',
		'posts_per_page' => 1,
		'fields'         => 'ids',
		'meta_query'     => array(
			array(
				'key'     => '_rpm_auto_posted',
				'value'   => '1',
				'compare' => '='
			)
		)
	));
	
	if ( $trashed_count->found_posts > 0 ) :
	?>
	<div style="margin-top: 20px; padding: 15px; background: #fff; border: 1px solid #c3c4c7; border-radius: 4px;">
		<form method="post" style="display: flex; align-items: center; gap: 15px;">
			<?php wp_nonce_field( 'rpm_empty_trash' ); ?>
			<div>
				<strong>🗑️ Trash:</strong> <?php echo number_format_i18n( $trashed_count->found_posts ); ?> auto-generated post(s) in trash
			</div>
			<button type="submit" name="rpm_empty_trash" value="1" class="button" onclick="return confirm('⚠️ This will PERMANENTLY DELETE <?php echo $trashed_count->found_posts; ?> post(s) from trash.\n\nThis action CANNOT be undone.\n\nAre you sure?');">
				🗑️ Empty Trash
			</button>
		</form>
	</div>
	<?php endif; ?>
	
	<hr style="margin: 30px 0;">
	
	<h2 class="title">Redistribute Posts</h2>
	<p>Redistribute posts across a date range to randomize publication times. Each month will have approximately the same number of posts (±15% variance).</p>
	
	<div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-bottom: 20px;">
		<div style="background: #f0f6fc; padding: 15px; border-left: 3px solid #2271b1; border-radius: 4px; margin-bottom: 15px;">
			<h3 style="margin-top: 0; font-size: 14px;">💡 How This Works</h3>
			<p style="margin: 0;">This tool will:</p>
			<ul style="margin: 10px 0 0 20px;">
				<li>Find all posts within your selected date range</li>
				<li>Redistribute them randomly across the date range</li>
				<li>Ensure each month has approximately the same number of posts (±15% variance)</li>
				<li>Update post dates while preserving post status</li>
			</ul>
		</div>
		
		<div style="display: flex; align-items: center; gap: 15px; margin-bottom: 15px; flex-wrap: wrap;">
			<div style="display: flex; align-items: center; gap: 8px;">
				<label for="rpm_redistribute_from" style="font-weight: 600;">From:</label>
				<input type="date" id="rpm_redistribute_from" class="regular-text" style="width: 150px;" />
			</div>
			<div style="display: flex; align-items: center; gap: 8px;">
				<label for="rpm_redistribute_to" style="font-weight: 600;">To:</label>
				<input type="date" id="rpm_redistribute_to" class="regular-text" style="width: 150px;" />
			</div>
			<button type="button" id="rpm_redistribute_posts" class="button button-primary" style="height: 40px;">📅 Redistribute Posts</button>
		</div>
		
		<div style="margin-bottom: 15px; padding: 10px; background: #f9f9f9; border-radius: 4px;">
			<label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
				<input type="checkbox" id="rpm_randomize_order" checked />
				<strong>🔀 Randomize post order before redistributing</strong>
			</label>
			<p class="description" style="margin: 5px 0 0 28px;">
				When enabled, posts will be shuffled randomly before being distributed across dates. 
				Uncheck to preserve original post order (oldest → newest).
			</p>
		</div>
		
		<p class="description" style="margin: 0 0 10px 0;">Select a date range to redistribute posts. Posts will be randomly distributed across this period.</p>
		
		<div id="rpm_redistribute_status" style="margin-top: 15px;"></div>
	</div>
</div>

<!-- Image picker modal is handled by includes/image-picker-modal.php -->

<script>
jQuery(document).ready(function($) {
	// Confirm permanent deletion
	$('form').on('submit', function(e) {
		var action = $('#bulk-action-selector-top').val() || $('#bulk-action-selector-bottom').val();
		
		if (action === 'delete') {
			var checkedCount = $('input[name="post[]"]:checked').length;
			if (checkedCount === 0) {
				return true;
			}
			
			if (!confirm('⚠️ WARNING: This will PERMANENTLY DELETE ' + checkedCount + ' post(s).\n\nThis action CANNOT be undone.\n\nAre you sure you want to continue?')) {
				e.preventDefault();
				return false;
			}
		}
	});
	
	// Handle post redistribution
	$('#rpm_redistribute_posts').on('click', function() {
		var fromDate = $('#rpm_redistribute_from').val();
		var toDate = $('#rpm_redistribute_to').val();
		var $button = $(this);
		
		if (!fromDate || !toDate) {
			$('#rpm_redistribute_status').html('<div class="notice notice-error"><p>❌ Please select both "From" and "To" dates.</p></div>');
			return;
		}
		
		if (new Date(fromDate) > new Date(toDate)) {
			$('#rpm_redistribute_status').html('<div class="notice notice-error"><p>❌ "From" date must be before "To" date.</p></div>');
			return;
		}
		
		if (!confirm('This will redistribute all posts between ' + fromDate + ' and ' + toDate + '. Continue?')) {
			return;
		}
		
		// Disable button and show processing message
		$button.prop('disabled', true).text('⏳ Redistributing...');
		$('#rpm_redistribute_status').html('<div class="notice notice-info"><p>⏳ <strong>Redistributing posts...</strong> This may take a moment.</p></div>');
		
		var randomizeOrder = $('#rpm_randomize_order').is(':checked');
		
		$.ajax({
			url: '/api/redistribute-posts.php',
			type: 'POST',
			data: {
				_wpnonce: '<?php echo wp_create_nonce( 'rpm_redistribute_posts' ); ?>',
				from_date: fromDate,
				to_date: toDate,
				randomize_order: randomizeOrder ? '1' : '0'
			},
			dataType: 'json',
			timeout: 60000, // 60 seconds
			success: function(response) {
				$button.prop('disabled', false).text('📅 Redistribute Posts');
				
				if (response.success) {
					var message = '✅ <strong>Success!</strong> Redistributed ' + response.posts_updated + ' post(s) across ' + response.months_count + ' month(s).';
					if (response.details) {
						message += '<br><small>' + response.details + '</small>';
					}
					$('#rpm_redistribute_status').html('<div class="notice notice-success"><p>' + message + '</p></div>');
				} else {
					$('#rpm_redistribute_status').html('<div class="notice notice-error"><p>❌ <strong>Error:</strong> ' + (response.message || 'Redistribution failed') + '</p></div>');
				}
			},
			error: function(xhr, status, error) {
				$button.prop('disabled', false).text('📅 Redistribute Posts');
				var errorMsg = 'Network error';
				
				if (xhr.responseJSON && xhr.responseJSON.message) {
					errorMsg = xhr.responseJSON.message;
				} else if (status === 'timeout') {
					errorMsg = 'Request timeout - please try again';
				}
				
				$('#rpm_redistribute_status').html('<div class="notice notice-error"><p>❌ <strong>Error:</strong> ' + errorMsg + '</p></div>');
			}
		});
	});
});
</script>

<style>
.rpm-stats-boxes {
	display: grid;
	grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
	gap: 20px;
	margin: 20px 0 30px 0;
}
.rpm-stat-box {
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 4px;
	padding: 20px;
	text-align: center;
	box-shadow: 0 1px 1px rgba(0,0,0,0.04);
}
.rpm-stat-number {
	font-size: 36px;
	font-weight: 600;
	color: #2271b1;
	margin-bottom: 5px;
}
.rpm-stat-label {
	font-size: 14px;
	color: #646970;
}
.rpm-info-footer {
	background: #f9f9f9;
	border: 1px solid #ddd;
	border-radius: 4px;
	padding: 20px;
	margin-top: 30px;
}
.rpm-info-footer h3 {
	margin-top: 0;
}
.rpm-info-footer ul {
	margin-left: 20px;
}
.rpm-info-footer code {
	background: #f0f0f0;
	padding: 2px 6px;
	border-radius: 3px;
}
.rpm-image-change-option:hover {
	transform: scale(1.02);
	box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}
</style>
