<?php
/**
 * Schema.org Structured Data and Open Graph Output
 * 
 * Outputs JSON-LD structured data and Open Graph/Twitter meta tags
 * for AI-generated posts to improve SEO and social sharing.
 * 
 * @package RemotePostManager
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Output structured data (JSON-LD) for single posts
 */
function rpm_output_structured_data() {
	// Only on single posts
	if ( ! is_single() ) {
		return;
	}
	
	$post_id = get_the_ID();
	
	// Check if this is an auto-generated post
	$is_auto_posted = get_post_meta( $post_id, '_rpm_auto_posted', true );
	
	// Get post data
	$post = get_post( $post_id );
	$title = get_the_title( $post_id );
	$excerpt = get_the_excerpt( $post_id );
	$permalink = get_permalink( $post_id );
	$date_published = get_the_date( 'c', $post_id );
	$date_modified = get_the_modified_date( 'c', $post_id );
	
	// Get author info
	$author_id = $post->post_author;
	$author_name = get_the_author_meta( 'display_name', $author_id );
	$author_url = get_author_posts_url( $author_id );
	
	// Get featured image
	$featured_image_url = '';
	if ( has_post_thumbnail( $post_id ) ) {
		$featured_image_url = get_the_post_thumbnail_url( $post_id, 'large' );
	}
	
	// Get site info
	$site_name = get_bloginfo( 'name' );
	$site_url = home_url();
	$site_logo = '';
	if ( has_custom_logo() ) {
		$logo_id = get_theme_mod( 'custom_logo' );
		$site_logo = wp_get_attachment_image_url( $logo_id, 'full' );
	}
	
	// Get SEO meta from post
	$focus_keyphrase = get_post_meta( $post_id, '_rpm_focus_keyphrase', true );
	$reading_time = get_post_meta( $post_id, '_rpm_reading_time', true );
	$word_count = get_post_meta( $post_id, '_rpm_word_count', true );
	
	// Initialize schemas array
	$schemas = array();
	
	// =========================================================================
	// ARTICLE SCHEMA (always output for single posts)
	// =========================================================================
	$article_schema = array(
		'@context'         => 'https://schema.org',
		'@type'            => 'Article',
		'headline'         => $title,
		'description'      => wp_strip_all_tags( $excerpt ),
		'url'              => $permalink,
		'datePublished'    => $date_published,
		'dateModified'     => $date_modified,
		'author'           => array(
			'@type' => 'Person',
			'name'  => $author_name,
			'url'   => $author_url,
		),
		'publisher'        => array(
			'@type' => 'Organization',
			'name'  => $site_name,
			'url'   => $site_url,
		),
		'mainEntityOfPage' => array(
			'@type' => 'WebPage',
			'@id'   => $permalink,
		),
	);
	
	// Add logo to publisher if available
	if ( ! empty( $site_logo ) ) {
		$article_schema['publisher']['logo'] = array(
			'@type' => 'ImageObject',
			'url'   => $site_logo,
		);
	}
	
	// Add featured image if available
	if ( ! empty( $featured_image_url ) ) {
		$article_schema['image'] = array(
			'@type' => 'ImageObject',
			'url'   => $featured_image_url,
		);
	}
	
	// Add word count if available
	if ( ! empty( $word_count ) ) {
		$article_schema['wordCount'] = (int) $word_count;
	}
	
	// Add keywords if focus keyphrase is set
	if ( ! empty( $focus_keyphrase ) ) {
		$article_schema['keywords'] = $focus_keyphrase;
	}
	
	$schemas[] = $article_schema;
	
	// =========================================================================
	// FAQ SCHEMA (if FAQ data exists)
	// =========================================================================
	$faq_json = get_post_meta( $post_id, '_rpm_faq_schema', true );
	if ( ! empty( $faq_json ) ) {
		$faq_data = json_decode( $faq_json, true );
		if ( is_array( $faq_data ) && ! empty( $faq_data ) ) {
			$faq_schema = array(
				'@context'   => 'https://schema.org',
				'@type'      => 'FAQPage',
				'mainEntity' => array(),
			);
			
			foreach ( $faq_data as $faq_item ) {
				if ( isset( $faq_item['question'] ) && isset( $faq_item['answer'] ) ) {
					$faq_schema['mainEntity'][] = array(
						'@type'          => 'Question',
						'name'           => $faq_item['question'],
						'acceptedAnswer' => array(
							'@type' => 'Answer',
							'text'  => $faq_item['answer'],
						),
					);
				}
			}
			
			if ( ! empty( $faq_schema['mainEntity'] ) ) {
				$schemas[] = $faq_schema;
			}
		}
	}
	
	// =========================================================================
	// HOWTO SCHEMA (if HowTo data exists and is_tutorial is true)
	// =========================================================================
	$howto_json = get_post_meta( $post_id, '_rpm_howto_schema', true );
	if ( ! empty( $howto_json ) ) {
		$howto_data = json_decode( $howto_json, true );
		if ( is_array( $howto_data ) && ! empty( $howto_data['is_tutorial'] ) && ! empty( $howto_data['steps'] ) ) {
			$howto_schema = array(
				'@context' => 'https://schema.org',
				'@type'    => 'HowTo',
				'name'     => $title,
				'step'     => array(),
			);
			
			// Add description
			if ( ! empty( $excerpt ) ) {
				$howto_schema['description'] = wp_strip_all_tags( $excerpt );
			}
			
			// Add image if available
			if ( ! empty( $featured_image_url ) ) {
				$howto_schema['image'] = $featured_image_url;
			}
			
			// Add estimated time if reading time available
			if ( ! empty( $reading_time ) ) {
				$howto_schema['totalTime'] = 'PT' . (int) $reading_time . 'M';
			}
			
			foreach ( $howto_data['steps'] as $index => $step ) {
				if ( isset( $step['name'] ) && isset( $step['text'] ) ) {
					$howto_schema['step'][] = array(
						'@type'    => 'HowToStep',
						'position' => $index + 1,
						'name'     => $step['name'],
						'text'     => $step['text'],
					);
				}
			}
			
			if ( ! empty( $howto_schema['step'] ) ) {
				$schemas[] = $howto_schema;
			}
		}
	}
	
	// =========================================================================
	// BREADCRUMB SCHEMA
	// =========================================================================
	$categories = get_the_category( $post_id );
	if ( ! empty( $categories ) ) {
		$primary_category = $categories[0];
		$breadcrumb_schema = array(
			'@context'        => 'https://schema.org',
			'@type'           => 'BreadcrumbList',
			'itemListElement' => array(
				array(
					'@type'    => 'ListItem',
					'position' => 1,
					'name'     => 'Home',
					'item'     => $site_url,
				),
				array(
					'@type'    => 'ListItem',
					'position' => 2,
					'name'     => $primary_category->name,
					'item'     => get_category_link( $primary_category->term_id ),
				),
				array(
					'@type'    => 'ListItem',
					'position' => 3,
					'name'     => $title,
					'item'     => $permalink,
				),
			),
		);
		$schemas[] = $breadcrumb_schema;
	}
	
	// =========================================================================
	// OUTPUT ALL SCHEMAS
	// =========================================================================
	if ( ! empty( $schemas ) ) {
		echo "\n<!-- FINDEXA AI Posts - Structured Data -->\n";
		foreach ( $schemas as $schema ) {
			echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>' . "\n";
		}
	}
}

/**
 * Output Open Graph and Twitter Card meta tags
 */
function rpm_output_social_meta_tags() {
	// Only on single posts
	if ( ! is_single() ) {
		return;
	}
	
	$post_id = get_the_ID();
	$post = get_post( $post_id );
	
	// Get post data
	$title = get_the_title( $post_id );
	$excerpt = get_the_excerpt( $post_id );
	$permalink = get_permalink( $post_id );
	
	// Get meta description if available
	$meta_description = wp_strip_all_tags( $excerpt );
	
	// Get featured image
	$featured_image_url = '';
	if ( has_post_thumbnail( $post_id ) ) {
		$featured_image_url = get_the_post_thumbnail_url( $post_id, 'large' );
	}
	
	// Get site info
	$site_name = get_bloginfo( 'name' );
	
	// Get author
	$author_name = get_the_author_meta( 'display_name', $post->post_author );
	
	// Get publish date
	$publish_date = get_the_date( 'c', $post_id );
	$modified_date = get_the_modified_date( 'c', $post_id );
	
	echo "\n<!-- FINDEXA AI Posts - Open Graph Meta -->\n";
	
	// =========================================================================
	// OPEN GRAPH TAGS
	// =========================================================================
	echo '<meta property="og:type" content="article" />' . "\n";
	echo '<meta property="og:title" content="' . esc_attr( $title ) . '" />' . "\n";
	echo '<meta property="og:description" content="' . esc_attr( $meta_description ) . '" />' . "\n";
	echo '<meta property="og:url" content="' . esc_url( $permalink ) . '" />' . "\n";
	echo '<meta property="og:site_name" content="' . esc_attr( $site_name ) . '" />' . "\n";
	
	if ( ! empty( $featured_image_url ) ) {
		echo '<meta property="og:image" content="' . esc_url( $featured_image_url ) . '" />' . "\n";
	}
	
	// Article-specific OG tags
	echo '<meta property="article:published_time" content="' . esc_attr( $publish_date ) . '" />' . "\n";
	echo '<meta property="article:modified_time" content="' . esc_attr( $modified_date ) . '" />' . "\n";
	echo '<meta property="article:author" content="' . esc_attr( $author_name ) . '" />' . "\n";
	
	// Add tags as article:tag
	$tags = get_the_tags( $post_id );
	if ( $tags ) {
		foreach ( $tags as $tag ) {
			echo '<meta property="article:tag" content="' . esc_attr( $tag->name ) . '" />' . "\n";
		}
	}
	
	// =========================================================================
	// TWITTER CARD TAGS
	// =========================================================================
	echo "\n<!-- FINDEXA AI Posts - Twitter Card Meta -->\n";
	echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
	echo '<meta name="twitter:title" content="' . esc_attr( $title ) . '" />' . "\n";
	echo '<meta name="twitter:description" content="' . esc_attr( $meta_description ) . '" />' . "\n";
	
	if ( ! empty( $featured_image_url ) ) {
		echo '<meta name="twitter:image" content="' . esc_url( $featured_image_url ) . '" />' . "\n";
	}
}

/**
 * Output reading time in post content (optional feature)
 * 
 * @param string $content Post content
 * @return string Modified content with reading time
 */
function rpm_add_reading_time_to_content( $content ) {
	if ( ! is_single() ) {
		return $content;
	}
	
	$post_id = get_the_ID();
	$reading_time = get_post_meta( $post_id, '_rpm_reading_time', true );
	
	if ( empty( $reading_time ) ) {
		return $content;
	}
	
	$reading_time_html = '<p class="rpm-reading-time" style="color: #666; font-size: 14px; margin-bottom: 20px;">';
	$reading_time_html .= '<span class="dashicons dashicons-clock" style="margin-right: 5px;"></span>';
	$reading_time_html .= sprintf( 
		/* translators: %d: reading time in minutes */
		__( '%d min read', 'remote-post-manager' ), 
		(int) $reading_time 
	);
	$reading_time_html .= '</p>';
	
	return $reading_time_html . $content;
}

/**
 * Register hooks for schema output
 */
function rpm_register_schema_hooks() {
	// Output structured data in wp_head
	add_action( 'wp_head', 'rpm_output_structured_data', 5 );
	add_action( 'wp_head', 'rpm_output_social_meta_tags', 5 );
	
	// Optionally add reading time to content (disabled by default)
	// add_filter( 'the_content', 'rpm_add_reading_time_to_content', 5 );
}
add_action( 'init', 'rpm_register_schema_hooks' );

