<?php
/**
 * Media Cleanup - Automatically delete attached media when posts are deleted
 * Only deletes media if it's not used by other posts
 *
 * @package RemotePostManager
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Delete attached media when a post is permanently deleted
 * Only deletes if the media is not used by other posts
 *
 * @param int $post_id The ID of the post being deleted
 */
function rpm_cleanup_post_media( $post_id ) {
    // Only run for posts (not pages, attachments, etc.)
    $post_type = get_post_type( $post_id );
    if ( $post_type !== 'post' ) {
        return;
    }
    
    // Get the featured image ID
    $thumbnail_id = get_post_thumbnail_id( $post_id );
    
    if ( $thumbnail_id ) {
        // Check if this image is used by any other posts
        if ( ! rpm_is_media_used_elsewhere( $thumbnail_id, $post_id ) ) {
            // Delete the attachment permanently
            wp_delete_attachment( $thumbnail_id, true );
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "RPM Media Cleanup: Deleted attachment ID {$thumbnail_id} (was attached to post {$post_id})" );
            }
        }
    }
    
    // Also check for any other attachments uploaded to this post
    $attachments = get_posts( array(
        'post_type'      => 'attachment',
        'posts_per_page' => -1,
        'post_parent'    => $post_id,
        'fields'         => 'ids',
    ) );
    
    foreach ( $attachments as $attachment_id ) {
        if ( ! rpm_is_media_used_elsewhere( $attachment_id, $post_id ) ) {
            wp_delete_attachment( $attachment_id, true );
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "RPM Media Cleanup: Deleted child attachment ID {$attachment_id} (was attached to post {$post_id})" );
            }
        }
    }
}
add_action( 'before_delete_post', 'rpm_cleanup_post_media' );

/**
 * Check if a media attachment is used by any other posts
 *
 * @param int $attachment_id The attachment ID to check
 * @param int $exclude_post_id The post ID to exclude from the check
 * @return bool True if used elsewhere, false if safe to delete
 */
function rpm_is_media_used_elsewhere( $attachment_id, $exclude_post_id ) {
    global $wpdb;
    
    // Check if used as featured image by any other post
    $featured_usage = $wpdb->get_var( $wpdb->prepare(
        "SELECT COUNT(*) FROM {$wpdb->postmeta} 
         WHERE meta_key = '_thumbnail_id' 
         AND meta_value = %d 
         AND post_id != %d",
        $attachment_id,
        $exclude_post_id
    ) );
    
    if ( $featured_usage > 0 ) {
        return true;
    }
    
    // Check if the image URL is referenced in any other post content
    $attachment_url = wp_get_attachment_url( $attachment_id );
    if ( $attachment_url ) {
        // Get just the filename part for more reliable matching
        $filename = basename( $attachment_url );
        
        $content_usage = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_type = 'post' 
             AND ID != %d 
             AND post_content LIKE %s",
            $exclude_post_id,
            '%' . $wpdb->esc_like( $filename ) . '%'
        ) );
        
        if ( $content_usage > 0 ) {
            return true;
        }
    }
    
    return false;
}

/**
 * Add admin notice about media cleanup feature
 */
function rpm_media_cleanup_admin_notice() {
    // Only show on our plugin pages
    $screen = get_current_screen();
    if ( ! $screen || strpos( $screen->id, 'remote-post-manager' ) === false ) {
        return;
    }
    
    // Check if we should show the notice (first time only)
    if ( get_option( 'rpm_media_cleanup_notice_dismissed' ) ) {
        return;
    }
    
    ?>
    <div class="notice notice-info is-dismissible" id="rpm-media-cleanup-notice">
        <p>
            <strong><?php esc_html_e( 'Media Cleanup Enabled', 'remote-post-manager' ); ?></strong><br>
            <?php esc_html_e( 'When you permanently delete posts, their featured images will also be deleted (if not used by other posts).', 'remote-post-manager' ); ?>
        </p>
    </div>
    <script>
    jQuery(document).on('click', '#rpm-media-cleanup-notice .notice-dismiss', function() {
        jQuery.post(ajaxurl, {
            action: 'rpm_dismiss_media_cleanup_notice',
            _wpnonce: '<?php echo wp_create_nonce( 'rpm_dismiss_notice' ); ?>'
        });
    });
    </script>
    <?php
}
add_action( 'admin_notices', 'rpm_media_cleanup_admin_notice' );

/**
 * AJAX handler to dismiss the media cleanup notice
 */
function rpm_dismiss_media_cleanup_notice() {
    check_ajax_referer( 'rpm_dismiss_notice' );
    update_option( 'rpm_media_cleanup_notice_dismissed', true );
    wp_die();
}
add_action( 'wp_ajax_rpm_dismiss_media_cleanup_notice', 'rpm_dismiss_media_cleanup_notice' );

