<?php
/**
 * Image Picker Modal
 * Shows alternate AI-selected images in a modal
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add button to post editor
 */
function rpm_add_image_picker_button() {
	global $post;
	if ( ! $post ) {
		return;
	}
	
	$image_options = get_post_meta( $post->ID, '_rpm_image_options', true );
	
	if ( empty( $image_options ) || ! is_array( $image_options ) ) {
		return;
	}
	
	echo '<div class="misc-pub-section">';
	echo '<button type="button" id="rpm-open-image-picker" class="button button-primary button-large" style="margin-top: 10px; width: 100%;">';
	echo '🖼️ View ' . count( $image_options ) . ' AI-Selected Images';
	echo '</button>';
	echo '</div>';
}
add_action( 'post_submitbox_misc_actions', 'rpm_add_image_picker_button' );

/**
 * Add "Change Image" link to posts list
 */
function rpm_add_image_column( $columns ) {
	$columns['rpm_image'] = 'Featured Image';
	return $columns;
}
add_filter( 'manage_posts_columns', 'rpm_add_image_column' );

function rpm_image_column_content( $column, $post_id ) {
	if ( $column === 'rpm_image' ) {
		$image_options = get_post_meta( $post_id, '_rpm_image_options', true );
		$thumbnail = get_the_post_thumbnail( $post_id, array( 50, 50 ) );
		
		if ( $thumbnail ) {
			echo $thumbnail . '<br>';
		}
		
		if ( ! empty( $image_options ) && is_array( $image_options ) ) {
			echo '<a href="#" class="rpm-change-image" data-post-id="' . $post_id . '" style="font-size: 11px;">Change (' . count( $image_options ) . ' options)</a>';
		}
	}
}
add_action( 'manage_posts_custom_column', 'rpm_image_column_content', 10, 2 );

/**
 * Enqueue modal scripts and styles
 */
function rpm_enqueue_image_picker() {
	global $post;
	$screen = get_current_screen();
	
	// Load on post editor, posts list, and custom Posts admin page
	$allowed_bases = array( 'post', 'edit' );
	$allowed_pages = array( 'remote-post-manager-auto-posts' );
	
	if ( ! $screen ) {
		return;
	}
	
	$is_allowed_base = in_array( $screen->base, $allowed_bases );
	$is_allowed_page = isset( $_GET['page'] ) && in_array( $_GET['page'], $allowed_pages );
	
	if ( ! $is_allowed_base && ! $is_allowed_page ) {
		return;
	}
	
	// Register and enqueue script with jQuery dependency
	wp_register_script( 'rpm-image-picker-js', false, array( 'jquery' ), '1.0', true );
	wp_enqueue_script( 'rpm-image-picker-js' );
	
	// Output CSS directly in admin footer
	add_action( 'admin_footer', function() {
		?>
		<style id="rpm-image-picker-styles">
		#rpm-image-modal {
			display: none;
			position: fixed !important;
			top: 0 !important;
			left: 0 !important;
			right: 0 !important;
			bottom: 0 !important;
			width: 100% !important;
			height: 100% !important;
			background: rgba(0,0,0,0.8) !important;
			z-index: 999999 !important;
			overflow-y: auto !important;
		}
		#rpm-image-modal .rpm-modal {
			background: white !important;
			max-width: 900px !important;
			margin: 50px auto !important;
			padding: 20px !important;
			border-radius: 8px !important;
			box-shadow: 0 5px 30px rgba(0,0,0,0.5) !important;
			position: relative !important;
		}
		#rpm-image-modal .rpm-modal-header {
			display: flex;
			justify-content: space-between;
			align-items: center;
			margin-bottom: 20px;
			border-bottom: 2px solid #0073aa;
			padding-bottom: 10px;
		}
		#rpm-image-modal .rpm-modal-close {
			font-size: 28px;
			cursor: pointer;
			color: #666;
			background: none;
			border: none;
			padding: 0 10px;
		}
		#rpm-image-modal .rpm-modal-close:hover {
			color: #d00;
		}
		#rpm-image-modal .rpm-image-grid {
			display: grid;
			grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
			gap: 15px;
		}
		#rpm-image-modal .rpm-image-option {
			border: 3px solid #ddd;
			padding: 10px;
			cursor: pointer;
			transition: all 0.2s;
			border-radius: 8px;
			background: #f9f9f9;
		}
		#rpm-image-modal .rpm-image-option:hover {
			border-color: #0073aa;
			box-shadow: 0 5px 15px rgba(0,115,170,0.3);
			transform: translateY(-3px);
		}
		#rpm-image-modal .rpm-image-option img {
			width: 100%;
			height: 150px;
			object-fit: cover;
			border-radius: 4px;
		}
		#rpm-image-modal .rpm-image-option p {
			margin: 8px 0 0 0;
			font-size: 11px;
			color: #666;
		}
		#rpm-image-modal .rpm-rank-badge {
			display: inline-block;
			background: #0073aa;
			color: white;
			padding: 3px 8px;
			border-radius: 4px;
			font-size: 10px;
			font-weight: bold;
			margin-bottom: 5px;
		}
		</style>
		<?php
	});
	
	// Prepare variables for JavaScript
	$current_post_id = isset( $post->ID ) ? $post->ID : 0;
	$nonce_get = wp_create_nonce( 'rpm_image_picker' );
	$nonce_set = wp_create_nonce( 'rpm_set_featured_image' );
	
	// Use heredoc for clean JavaScript
	$inline_script = <<<ENDJS
jQuery(document).ready(function($) {
	console.log('RPM Image Picker: Script loaded');
	console.log('RPM Image Picker: jQuery version:', $.fn.jquery);
	var currentPostId = {$current_post_id};
	console.log('RPM Image Picker: Current post ID:', currentPostId);
	
	// Alert to confirm script is running
	console.log('RPM Image Picker: About to create modal HTML');
	
	// Create modal HTML
	var existingModal = $('#rpm-image-modal').length;
	console.log('RPM Image Picker: Existing modal count:', existingModal);
	
	if (existingModal === 0) {
		console.log('RPM Image Picker: Creating modal HTML');
		$('body').append(
			'<div id="rpm-image-modal" class="rpm-modal-overlay">' +
				'<div class="rpm-modal">' +
					'<div class="rpm-modal-header">' +
						'<h2>🤖 AI-Selected Images (Ranked by Vision AI)</h2>' +
						'<button class="rpm-modal-close">&times;</button>' +
					'</div>' +
					'<div id="rpm-image-grid" class="rpm-image-grid"></div>' +
				'</div>' +
			'</div>'
		);
		console.log('RPM Image Picker: Modal HTML appended, checking...');
		console.log('RPM Image Picker: Modal now exists?', $('#rpm-image-modal').length > 0);
	} else {
		console.log('RPM Image Picker: Modal already exists');
	}
	
	// Count click handlers being registered
	console.log('RPM Image Picker: Registering click handlers');
	
	// Open modal from edit post page
	$(document).on('click', '#rpm-open-image-picker', function(e) {
		console.log('RPM Image Picker: Button clicked (edit page)');
		e.preventDefault();
		loadImages(currentPostId);
	});
	
	// Open modal from posts list
	$(document).on('click', '.rpm-change-image', function(e) {
		console.log('RPM Image Picker: Change image link clicked');
		e.preventDefault();
		var postId = $(this).data('post-id');
		console.log('RPM Image Picker: Post ID from link:', postId);
		loadImages(postId);
	});
	
	// Close modal
	$(document).on('click', '.rpm-modal-close, .rpm-modal-overlay', function(e) {
		if (e.target === this) {
			$('#rpm-image-modal').fadeOut(200);
		}
	});
	
	// Load images via AJAX
	function loadImages(postId) {
		console.log('RPM Image Picker: Loading images for post', postId);
		
		if (!postId || postId === 0) {
			alert('Invalid post ID: ' + postId);
			return;
		}
		
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'rpm_get_image_options',
				post_id: postId,
				nonce: '{$nonce_get}'
			},
		success: function(response) {
			console.log('RPM Image Picker: AJAX response:', response);
			if (response.success && response.data.images) {
				displayImages(response.data.images, postId);
				console.log('RPM Image Picker: About to show modal');
				console.log('RPM Image Picker: Modal element exists?', $('#rpm-image-modal').length);
				console.log('RPM Image Picker: Modal display before:', $('#rpm-image-modal').css('display'));
				console.log('RPM Image Picker: Modal z-index:', $('#rpm-image-modal').css('z-index'));
				// Force display and show
				$('#rpm-image-modal').css('display', 'block').show();
				setTimeout(function() {
					console.log('RPM Image Picker: Modal display after:', $('#rpm-image-modal').css('display'));
					console.log('RPM Image Picker: Modal visible?', $('#rpm-image-modal').is(':visible'));
				}, 300);
			} else {
				alert('No alternate images available for this post.');
			}
		},
			error: function(xhr, status, error) {
				console.error('RPM Image Picker: AJAX error:', status, error);
				alert('Error loading images: ' + error);
			}
		});
	}
	
	// Display images in modal
	function displayImages(images, postId) {
		console.log('RPM Image Picker: Displaying', images.length, 'images');
		var html = '';
		images.forEach(function(img, index) {
			var imgUrl = img.url || '';
			var imgMedium = img.url_medium || img.url || '';
			var altText = img.alt_text || '';
			var photographer = img.photographer || 'Unknown';
			var pexelsId = img.pexels_id || 0;
			
			html += '<div class="rpm-image-option" data-post-id="' + postId + '" data-image-url="' + imgUrl + '" data-photographer="' + photographer + '" data-pexels-id="' + pexelsId + '">';
			html += '<div class="rpm-rank-badge">#' + (index + 1) + ' AI Ranked</div>';
			html += '<img src="' + imgMedium + '" alt="' + altText + '">';
			html += '<p>📸 ' + photographer + '</p>';
			html += '</div>';
		});
		$('#rpm-image-grid').html(html);
	}
	
	// Select image
	$(document).on('click', '.rpm-image-option', function() {
		console.log('RPM Image Picker: Image clicked');
		var \$this = $(this);
		var postId = \$this.data('post-id');
		var imageUrl = \$this.data('image-url');
		var photographer = \$this.data('photographer');
		var pexelsId = \$this.data('pexels-id');
		
		console.log('RPM Image Picker: Setting image', imageUrl, 'for post', postId);
		
		\$this.html('<div style="padding: 40px; text-align: center;">⏳ Setting image...</div>');
		
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'rpm_set_featured_image',
				post_id: postId,
				image_url: imageUrl,
				photographer: photographer,
				pexels_id: pexelsId,
				nonce: '{$nonce_set}'
			},
			success: function(response) {
				console.log('RPM Image Picker: Set image response:', response);
				if (response.success) {
					\$this.html('<div style="padding: 40px; text-align: center; color: #0a0;">✓ Image Set!</div>');
					setTimeout(function() {
						location.reload();
					}, 1000);
				} else {
					alert('Error: ' + (response.data.message || 'Failed to set image'));
					location.reload();
				}
			},
			error: function(xhr, status, error) {
				console.error('RPM Image Picker: Set image error:', status, error);
				alert('Error setting image: ' + error);
				location.reload();
			}
		});
	});
});
ENDJS;
	
	wp_add_inline_script( 'rpm-image-picker-js', $inline_script );
}
add_action( 'admin_enqueue_scripts', 'rpm_enqueue_image_picker' );

/**
 * AJAX: Get image options for a post
 */
function rpm_ajax_get_image_options() {
	check_ajax_referer( 'rpm_image_picker', 'nonce' );
	
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error( array( 'message' => 'Insufficient permissions' ) );
	}
	
	$post_id = intval( $_POST['post_id'] ?? 0 );
	
	if ( ! $post_id ) {
		wp_send_json_error( array( 'message' => 'Invalid post ID' ) );
	}
	
	$image_options = get_post_meta( $post_id, '_rpm_image_options', true );
	
	if ( empty( $image_options ) || ! is_array( $image_options ) ) {
		wp_send_json_error( array( 'message' => 'No images available' ) );
	}
	
	wp_send_json_success( array( 'images' => $image_options ) );
}
add_action( 'wp_ajax_rpm_get_image_options', 'rpm_ajax_get_image_options' );

/**
 * AJAX: Set featured image
 */
function rpm_ajax_set_featured_image() {
	check_ajax_referer( 'rpm_set_featured_image', 'nonce' );
	
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error( array( 'message' => 'Insufficient permissions' ) );
	}
	
	$post_id = intval( $_POST['post_id'] ?? 0 );
	$image_url = esc_url_raw( $_POST['image_url'] ?? '' );
	
	if ( ! $post_id || ! $image_url ) {
		wp_send_json_error( array( 'message' => 'Invalid parameters' ) );
	}
	
	// Get photographer and pexels_id from POST or from stored image options
	$photographer = sanitize_text_field( $_POST['photographer'] ?? '' );
	$pexels_id = intval( $_POST['pexels_id'] ?? 0 );
	
	// If not provided, try to get from stored image options
	if ( empty( $photographer ) || empty( $pexels_id ) ) {
		$image_options = get_post_meta( $post_id, '_rpm_image_options', true );
		if ( is_array( $image_options ) ) {
			foreach ( $image_options as $img ) {
				if ( isset( $img['url'] ) && $img['url'] === $image_url ) {
					if ( empty( $photographer ) && ! empty( $img['photographer'] ) ) {
						$photographer = $img['photographer'];
					}
					if ( empty( $pexels_id ) && ! empty( $img['pexels_id'] ) ) {
						$pexels_id = intval( $img['pexels_id'] );
					}
					break;
				}
			}
		}
	}
	
	// Download and attach image
	$image_id = rpm_download_and_attach_image(
		$image_url,
		$post_id,
		array(
			'photographer' => $photographer,
			'pexels_id' => $pexels_id,
		)
	);
	
	if ( is_wp_error( $image_id ) ) {
		wp_send_json_error( array( 'message' => $image_id->get_error_message() ) );
	}
	
	// Set as featured image
	set_post_thumbnail( $post_id, $image_id );
	
	wp_send_json_success( array( 'image_id' => $image_id ) );
}
add_action( 'wp_ajax_rpm_set_featured_image', 'rpm_ajax_set_featured_image' );
