<?php
/**
 * Image Selection Meta Box
 * Shows alternate AI-selected images for posts
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register the meta box
 */
function rpm_add_image_options_meta_box() {
	add_meta_box(
		'rpm_image_options',
		'AI Generated Image Options',
		'rpm_image_options_meta_box_callback',
		'post',
		'side',
		'default'
	);
}
add_action( 'add_meta_boxes', 'rpm_add_image_options_meta_box' );

/**
 * Render the meta box content
 */
function rpm_image_options_meta_box_callback( $post ) {
	$image_options = get_post_meta( $post->ID, '_rpm_image_options', true );
	
	if ( empty( $image_options ) || ! is_array( $image_options ) ) {
		echo '<p>No alternate images available.</p>';
		echo '<p><em>Generate a new post with AI to see image options.</em></p>';
		return;
	}
	
	wp_nonce_field( 'rpm_set_featured_image', 'rpm_image_nonce' );
	
	echo '<div class="rpm-image-options" style="max-height: 400px; overflow-y: auto;">';
	echo '<p><strong>' . count( $image_options ) . ' AI-selected images:</strong></p>';
	
	foreach ( $image_options as $index => $image ) {
		$url = esc_url( $image['url_medium'] ?? $image['url'] );
		$photographer = esc_html( $image['photographer'] ?? 'Unknown' );
		$pexels_id = intval( $image['pexels_id'] ?? 0 );
		
		echo '<div class="rpm-image-option" style="margin-bottom: 15px; padding: 10px; border: 1px solid #ddd; border-radius: 4px;">';
		echo '<img src="' . $url . '" style="width: 100%; height: auto; cursor: pointer; border-radius: 3px;" />';
		echo '<p style="margin: 5px 0 0 0; font-size: 11px; color: #666;">Photo by ' . $photographer . '</p>';
		echo '<button type="button" class="button button-small rpm-set-featured-btn" data-post-id="' . $post->ID . '" data-image-url="' . esc_attr( $image['url'] ) . '" data-photographer="' . $photographer . '" data-pexels-id="' . $pexels_id . '" style="margin-top: 5px; width: 100%;">Set as Featured Image</button>';
		echo '</div>';
	}
	
	echo '</div>';
	
	// Add JavaScript
	?>
	<script>
	jQuery(document).ready(function($) {
		$('.rpm-set-featured-btn').on('click', function(e) {
			e.preventDefault();
			var btn = $(this);
			btn.prop('disabled', true).text('Setting...');
			
			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'rpm_set_featured_image',
					post_id: btn.data('post-id'),
					image_url: btn.data('image-url'),
					photographer: btn.data('photographer'),
					pexels_id: btn.data('pexels-id'),
					nonce: $('#rpm_image_nonce').val()
				},
				success: function(response) {
					if (response.success) {
						btn.text('✓ Set!').css('background', '#46b450');
						setTimeout(function() { location.reload(); }, 1000);
					} else {
						alert('Error: ' + (response.data.message || 'Failed'));
						btn.prop('disabled', false).text('Set as Featured Image');
					}
				},
				error: function() {
					alert('Error setting featured image');
					btn.prop('disabled', false).text('Set as Featured Image');
				}
			});
		});
	});
	</script>
	<?php
}

/**
 * AJAX handler to set featured image
 */
function rpm_ajax_set_featured_image() {
	check_ajax_referer( 'rpm_set_featured_image', 'nonce' );
	
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error( array( 'message' => 'Insufficient permissions' ) );
	}
	
	$post_id = intval( $_POST['post_id'] ?? 0 );
	$image_url = esc_url_raw( $_POST['image_url'] ?? '' );
	
	if ( ! $post_id || ! $image_url ) {
		wp_send_json_error( array( 'message' => 'Invalid parameters' ) );
	}
	
	// Download and attach image
	$image_id = rpm_download_and_attach_image(
		$image_url,
		$post_id,
		array(
			'photographer' => sanitize_text_field( $_POST['photographer'] ?? '' ),
			'pexels_id' => intval( $_POST['pexels_id'] ?? 0 ),
		)
	);
	
	if ( is_wp_error( $image_id ) ) {
		wp_send_json_error( array( 'message' => $image_id->get_error_message() ) );
	}
	
	// Set as featured image
	set_post_thumbnail( $post_id, $image_id );
	
	wp_send_json_success( array( 'image_id' => $image_id ) );
}
add_action( 'wp_ajax_rpm_set_featured_image', 'rpm_ajax_set_featured_image' );
