<?php
/**
 * WP-Cron Scheduler for Automated Daily Posting
 * 
 * Uses user-selected hour + random minute offset to distribute load
 */

defined( 'ABSPATH' ) || exit;

// Define the cron hook name
define( 'RPM_CRON_HOOK', 'rpm_daily_auto_post' );

/**
 * Register the cron settings
 */
function rpm_register_cron_settings() {
    register_setting(
        'rpm_daily_limit_group',
        'rpm_auto_post_hour',
        array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 9, // Default 9 AM
        )
    );
}
add_action( 'admin_init', 'rpm_register_cron_settings' );

/**
 * Calculate the random minute offset based on API key
 * This ensures each site posts at a consistent but distributed time
 */
function rpm_get_minute_offset() {
    $api_key = get_option( 'rpm_api_key', '' );
    if ( empty( $api_key ) ) {
        // Fallback to site URL hash if no API key
        $api_key = home_url();
    }
    // Use hash to get a consistent 0-59 minute offset
    return hexdec( substr( md5( $api_key ), 0, 2 ) ) % 60;
}

/**
 * Schedule or reschedule the daily cron event
 */
function rpm_schedule_daily_post() {
    $enabled = get_option( 'rpm_auto_posting_enabled', false );
    
    // Clear existing schedule first
    $timestamp = wp_next_scheduled( RPM_CRON_HOOK );
    if ( $timestamp ) {
        wp_unschedule_event( $timestamp, RPM_CRON_HOOK );
    }
    
    if ( ! $enabled ) {
        return; // Auto-posting disabled, don't schedule
    }
    
    // Get user's preferred hour and our random minute offset
    $hour = get_option( 'rpm_auto_post_hour', 9 );
    $minute = rpm_get_minute_offset();
    
    // Calculate next run time
    $timezone = wp_timezone();
    $now = new DateTime( 'now', $timezone );
    $scheduled = new DateTime( 'today', $timezone );
    $scheduled->setTime( $hour, $minute, 0 );
    
    // If the time has passed today, schedule for tomorrow
    if ( $scheduled <= $now ) {
        $scheduled->modify( '+1 day' );
    }
    
    // Schedule the event
    wp_schedule_event( $scheduled->getTimestamp(), 'daily', RPM_CRON_HOOK );
    
    // Log for debugging
    error_log( sprintf(
        '[FINDEXA] Scheduled daily post for %s (hour: %d, minute offset: %d)',
        $scheduled->format( 'Y-m-d H:i:s T' ),
        $hour,
        $minute
    ) );
}

/**
 * Reschedule when settings are updated
 */
function rpm_on_settings_update( $option, $old_value, $value ) {
    if ( in_array( $option, array( 'rpm_auto_posting_enabled', 'rpm_auto_post_hour' ), true ) ) {
        rpm_schedule_daily_post();
    }
}
add_action( 'updated_option', 'rpm_on_settings_update', 10, 3 );

/**
 * The actual cron callback - triggers the daily post generation
 */
function rpm_execute_daily_post() {
    // Check if auto-posting is still enabled
    if ( ! get_option( 'rpm_auto_posting_enabled', false ) ) {
        error_log( '[FINDEXA] Daily cron fired but auto-posting is disabled' );
        return;
    }
    
    $posts_per_day = get_option( 'rpm_posts_per_day', 1 );
    
    error_log( sprintf( '[FINDEXA] Executing daily auto-post: %d posts', $posts_per_day ) );
    
    // Call the AJAX handler function directly (simulating an admin request)
    if ( function_exists( 'rpm_generate_daily_posts' ) ) {
        $result = rpm_generate_daily_posts( $posts_per_day );
        error_log( '[FINDEXA] Daily post result: ' . print_r( $result, true ) );
    } else {
        // Fallback: Make internal request to AJAX endpoint
        $response = wp_remote_post( admin_url( 'admin-ajax.php' ), array(
            'body' => array(
                'action' => 'rpm_trigger_daily_post',
                'posts_count' => $posts_per_day,
                'is_cron' => true,
                '_wpnonce' => wp_create_nonce( 'rpm_trigger_daily_post' ),
            ),
            'timeout' => 120,
            'sslverify' => false,
        ) );
        
        if ( is_wp_error( $response ) ) {
            error_log( '[FINDEXA] Daily post error: ' . $response->get_error_message() );
        }
    }
}
add_action( RPM_CRON_HOOK, 'rpm_execute_daily_post' );

/**
 * Plugin activation - schedule the cron
 */
function rpm_cron_activation() {
    rpm_schedule_daily_post();
}

/**
 * Plugin deactivation - clear the cron
 */
function rpm_cron_deactivation() {
    $timestamp = wp_next_scheduled( RPM_CRON_HOOK );
    if ( $timestamp ) {
        wp_unschedule_event( $timestamp, RPM_CRON_HOOK );
    }
}

/**
 * Get the next scheduled run time for display
 */
function rpm_get_next_scheduled_time() {
    $timestamp = wp_next_scheduled( RPM_CRON_HOOK );
    if ( ! $timestamp ) {
        return null;
    }
    
    $timezone = wp_timezone();
    $dt = new DateTime( '@' . $timestamp );
    $dt->setTimezone( $timezone );
    
    return $dt->format( 'Y-m-d H:i:s T' );
}

/**
 * Get current scheduling info for admin display
 */
function rpm_get_cron_info() {
    $enabled = get_option( 'rpm_auto_posting_enabled', false );
    $hour = get_option( 'rpm_auto_post_hour', 9 );
    $minute = rpm_get_minute_offset();
    $next_run = rpm_get_next_scheduled_time();
    
    return array(
        'enabled' => $enabled,
        'hour' => $hour,
        'minute_offset' => $minute,
        'scheduled_time' => sprintf( '%02d:%02d', $hour, $minute ),
        'next_run' => $next_run,
    );
}

