<?php
/**
 * FastAPI Client Class
 * 
 * Handles communication with FastAPI AI content generation service
 * 
 * @package RemotePostManager
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class RPM_FastAPI_Client {

	/**
	 * FastAPI base URL
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * FastAPI Bearer token
	 *
	 * @var string
	 */
	private $api_key;

	/**
	 * Request timeout in seconds
	 *
	 * @var int
	 */
	private $timeout = 60;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->api_url = get_option( 'rpm_fastapi_url', 'https://findexa.ca/api' );
		$this->api_key = get_option( 'rpm_fastapi_key', '' );
	}

	/**
	 * Generate content using FastAPI
	 *
	 * @param string $topic Blog post topic
	 * @param array  $snapshot Settings snapshot
	 * @param bool   $is_daily_auto_post Whether this is from Daily Auto-Post trigger (default: false)
	 * @return array|WP_Error Generated content or error
	 */
	public function generate_content( $topic, $snapshot = array(), $is_daily_auto_post = false ) {
		// Validate configuration
		if ( empty( $this->api_url ) || empty( $this->api_key ) ) {
			return new WP_Error(
				'fastapi_not_configured',
				'FastAPI URL or API key not configured. Please configure in Settings.',
				array( 'status' => 500 )
			);
		}

		// Build request payload
		$built_snapshot = $this->build_snapshot( $snapshot );
		
		// DEBUG: Log what we're sending to FastAPI
		error_log( 'DEBUG CLIENT - Built snapshot keys: ' . implode( ', ', array_keys( $built_snapshot ) ) );
		error_log( 'DEBUG CLIENT - site_description length: ' . strlen( $built_snapshot['site_description'] ?? '' ) );
		error_log( 'DEBUG CLIENT - category_name: ' . ( $built_snapshot['category_name'] ?? 'NOT SET' ) );
		
		$payload = array(
			'topic'    => $topic,
			'snapshot' => $built_snapshot,
		);

		// Add query parameter for daily auto-post tracking
		$endpoint = '/api/generate-content';
		if ( $is_daily_auto_post ) {
			$endpoint .= '?is_daily_auto_post=true';
		}

		// Make request
		return $this->make_request( $endpoint, $payload );
	}

	/**
	 * Generate a blog post topic using FastAPI
	 *
	 * @param array $snapshot Settings snapshot
	 * @return array|WP_Error Topic result or error
	 */
	public function generate_topic( $snapshot = array() ) {
		// Validate configuration
		if ( empty( $this->api_url ) || empty( $this->api_key ) ) {
			return new WP_Error(
				'fastapi_not_configured',
				'FastAPI URL or API key not configured. Please configure in Settings.',
				array( 'status' => 500 )
			);
		}

		// Build request payload
		$built_snapshot = $this->build_snapshot( $snapshot );

		// Make request to topic generation endpoint
		return $this->make_request( '/api/generate-topic', $built_snapshot );
	}

	/**
	 * Test connection to FastAPI
	 *
	 * @return array|WP_Error Connection test result or error
	 */
	public function test_connection() {
		if ( empty( $this->api_url ) || empty( $this->api_key ) ) {
			return new WP_Error(
				'fastapi_not_configured',
				'FastAPI URL or API key not configured',
				array( 'status' => 500 )
			);
		}

		// Simple validation - just check if API key and URL are set
		// The actual connection test happens when content is generated
		return array(
			'success' => true,
			'user'    => array(
				'username'    => 'api_user',
				'email'       => 'configured@findexa.ca',
				'permissions' => array(
					'can_create_posts' => true,
				),
			),
		);
	}

	/**
	 * Build settings snapshot
	 *
	 * @param array $custom_settings Custom settings to merge
	 * @return array Complete snapshot
	 */
	private function build_snapshot( $custom_settings = array() ) {
		// DEBUG: Write to file
		file_put_contents( '/tmp/rpm_debug.log', date( 'Y-m-d H:i:s' ) . " - BUILD_SNAPSHOT INPUT\n", FILE_APPEND );
		file_put_contents( '/tmp/rpm_debug.log', 'Input keys: ' . implode( ', ', array_keys( $custom_settings ) ) . "\n", FILE_APPEND );
		file_put_contents( '/tmp/rpm_debug.log', 'Has site_description: ' . ( isset( $custom_settings['site_description'] ) ? 'YES' : 'NO' ) . "\n", FILE_APPEND );
		file_put_contents( '/tmp/rpm_debug.log', 'Has category_name: ' . ( isset( $custom_settings['category_name'] ) ? 'YES' : 'NO' ) . "\n\n", FILE_APPEND );
		
		// Start with custom settings (from AJAX) - these should have ALL fields
		// Only add defaults for missing fields
		$snapshot = $custom_settings;
		
		// Add defaults only if not provided
		if ( ! isset( $snapshot['tone'] ) ) {
			$snapshot['tone'] = get_option( 'rpm_tone', 'professional' );
		}
		if ( ! isset( $snapshot['writing_style'] ) ) {
			$snapshot['writing_style'] = get_option( 'rpm_writing_style', 'informative' );
		}
		if ( ! isset( $snapshot['word_count'] ) ) {
			$snapshot['word_count'] = (int) get_option( 'rpm_word_count', 1500 );
		}
		if ( ! isset( $snapshot['target_audience'] ) ) {
			$snapshot['target_audience'] = get_option( 'rpm_target_audience', 'general readers' );
		}
		if ( ! isset( $snapshot['seo_keywords'] ) ) {
			$snapshot['seo_keywords'] = array();
		}
		if ( ! isset( $snapshot['categories'] ) ) {
			$snapshot['categories'] = array();
		}
		if ( ! isset( $snapshot['tags'] ) ) {
			$snapshot['tags'] = array();
		}
		if ( ! isset( $snapshot['include_images'] ) ) {
			$snapshot['include_images'] = true;
		}
		if ( ! isset( $snapshot['image_query'] ) ) {
			$snapshot['image_query'] = null;
		}
		
		// Return snapshot with ALL custom fields preserved
		return $snapshot;
	}

	/**
	 * Make HTTP request to FastAPI
	 *
	 * @param string $endpoint API endpoint (e.g., '/api/generate-content')
	 * @param array  $data Request data
	 * @param string $method HTTP method (GET, POST, PUT, DELETE)
	 * @return array|WP_Error Response or error
	 */
	private function make_request( $endpoint, $data = null, $method = 'POST' ) {
		$url = rtrim( $this->api_url, '/' ) . $endpoint;

		// Build request arguments
		$args = array(
			'method'  => $method,
			'timeout' => $this->timeout,
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->api_key,
				'Content-Type'  => 'application/json',
			),
		);

		// Add body for POST/PUT requests
		if ( in_array( $method, array( 'POST', 'PUT' ), true ) && ! empty( $data ) ) {
			$args['body'] = wp_json_encode( $data );
			
			// DEBUG: Log what we're sending
			file_put_contents( '/tmp/rpm_debug.log', date( 'Y-m-d H:i:s' ) . " - MAKE_REQUEST TO FASTAPI\n", FILE_APPEND );
			file_put_contents( '/tmp/rpm_debug.log', 'Endpoint: ' . $endpoint . "\n", FILE_APPEND );
			if ( isset( $data['snapshot'] ) ) {
				file_put_contents( '/tmp/rpm_debug.log', 'Snapshot keys in payload: ' . implode( ', ', array_keys( $data['snapshot'] ) ) . "\n", FILE_APPEND );
			}
			file_put_contents( '/tmp/rpm_debug.log', 'JSON body length: ' . strlen( $args['body'] ) . " bytes\n\n", FILE_APPEND );
		}

		// Make request
		$response = wp_remote_request( $url, $args );

		// Handle errors
		if ( is_wp_error( $response ) ) {
			return new WP_Error(
				'fastapi_request_failed',
				'Failed to connect to FastAPI: ' . $response->get_error_message(),
				array( 'status' => 500 )
			);
		}

		// Get response code and body
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		// Parse JSON response
		$json_response = json_decode( $response_body, true );

		// Handle non-200 responses
		if ( $response_code !== 200 ) {
			$error_message = 'FastAPI error';

			if ( isset( $json_response['detail'] ) ) {
				$error_message = $json_response['detail'];
			} elseif ( isset( $json_response['message'] ) ) {
				$error_message = $json_response['message'];
			}

			return new WP_Error(
				'fastapi_error',
				$error_message,
				array(
					'status'        => $response_code,
					'response_body' => $response_body,
				)
			);
		}

		// Return parsed response
		return $json_response;
	}

	/**
	 * Get API configuration status
	 *
	 * @return array Configuration status
	 */
	public function get_config_status() {
		return array(
			'configured'   => ! empty( $this->api_url ) && ! empty( $this->api_key ),
			'api_url'      => $this->api_url,
			'has_api_key'  => ! empty( $this->api_key ),
		);
	}

	/**
	 * Set custom timeout
	 *
	 * @param int $timeout Timeout in seconds
	 */
	public function set_timeout( $timeout ) {
		$this->timeout = (int) $timeout;
	}

	/**
	 * Suggest tags based on site content
	 *
	 * @param array $pages Array of page content to analyze
	 * @param int   $count Number of tags to suggest (default: 10)
	 * @return array|WP_Error Suggested tags or error
	 */
	public function suggest_tags( $pages, $count = 10 ) {
		if ( empty( $this->api_url ) || empty( $this->api_key ) ) {
			return new WP_Error(
				'fastapi_not_configured',
				'FastAPI URL or API key not configured',
				array( 'status' => 500 )
			);
		}

		$payload = array(
			'pages' => $pages,
			'count' => $count,
		);

		return $this->make_request( '/suggest-tags', $payload );
	}

	/**
	 * Suggest categories based on site content
	 *
	 * @param array $pages Array of page content to analyze
	 * @param int   $count Number of categories to suggest (default: 5)
	 * @return array|WP_Error Suggested categories or error
	 */
	public function suggest_categories( $pages, $count = 5 ) {
		if ( empty( $this->api_url ) || empty( $this->api_key ) ) {
			return new WP_Error(
				'fastapi_not_configured',
				'FastAPI URL or API key not configured',
				array( 'status' => 500 )
			);
		}

		$payload = array(
			'pages' => $pages,
			'count' => $count,
		);

		return $this->make_request( '/suggest-categories', $payload );
	}

}
