<?php
/**
 * AJAX Handlers for Remote Post Manager
 * 
 * Handles AJAX requests for AI content generation
 * 
 * @package RemotePostManager
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load FastAPI client if not already loaded
require_once RPM_PLUGIN_DIR . 'includes/class-fastapi-client.php';

/**
 * AJAX handler to generate a single post
 */
function rpm_ajax_generate_single_post() {
	// Verify nonce
	check_ajax_referer( 'rpm_generate_post', 'nonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}

	// Get request data
	$topic        = isset( $_POST['topic'] ) ? sanitize_text_field( $_POST['topic'] ) : '';
	$topic_raw    = $_POST['topic'] ?? '';
	// DEBUG: Log what we received
	error_log( 'RPM AJAX - Received topic type: ' . gettype( $topic_raw ) );
	error_log( 'RPM AJAX - Received topic value: ' . print_r( $topic_raw, true ) );
	error_log( 'RPM AJAX - Sanitized topic: ' . $topic );
	$topic_id     = isset( $_POST['topic_id'] ) ? sanitize_text_field( $_POST['topic_id'] ) : '';
	$snapshot_raw = isset( $_POST['snapshot'] ) ? $_POST['snapshot'] : array();

	// Validate topic
	if ( empty( $topic ) ) {
		wp_send_json_error(
			array(
				'message' => 'Topic is required',
			),
			400
		);
	}

	// Sanitize snapshot
	$snapshot = rpm_sanitize_snapshot( $snapshot_raw );
	
	// DEBUG: Write to file
	file_put_contents( '/tmp/rpm_debug.log', date( 'Y-m-d H:i:s' ) . " - AJAX HANDLER\n", FILE_APPEND );
	file_put_contents( '/tmp/rpm_debug.log', 'Sanitized keys: ' . implode( ', ', array_keys( $snapshot ) ) . "\n", FILE_APPEND );
	file_put_contents( '/tmp/rpm_debug.log', 'site_description: ' . ( $snapshot['site_description'] ?? 'MISSING' ) . "\n", FILE_APPEND );
	file_put_contents( '/tmp/rpm_debug.log', 'category_name: ' . ( $snapshot['category_name'] ?? 'MISSING' ) . "\n", FILE_APPEND );
	file_put_contents( '/tmp/rpm_debug.log', 'brand_keywords: ' . ( $snapshot['brand_keywords'] ?? 'MISSING' ) . "\n\n", FILE_APPEND );
	
	// category_name already comes from JavaScript (no need to convert here)

	// Add excluded Pexels IDs to prevent re-using images
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$snapshot['exclude_pexels_ids'] = $used_pexels_ids;

	// Add existing tags for AI to choose from (if not already in snapshot)
	if ( empty( $snapshot['tags'] ) ) {
		$existing_tags = get_tags( array( 'hide_empty' => false, 'orderby' => 'count', 'order' => 'DESC' ) );
		$tag_names = array();
		foreach ( $existing_tags as $tag ) {
			$tag_names[] = $tag->name;
		}
		$snapshot['tags'] = $tag_names;
	}

	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();

	// Generate content
	$result = $fastapi_client->generate_content( $topic, $snapshot );

	// Check for errors
	if ( is_wp_error( $result ) ) {
		wp_send_json_error(
			array(
				'message' => $result->get_error_message(),
			),
			500
		);
	}

	// Validate response
	if ( ! isset( $result['success'] ) || ! $result['success'] ) {
		wp_send_json_error(
			array(
				'message' => 'Failed to generate content',
			),
			500
		);
	}

	// Extract generated content
	$generated_content = $result['generated_content'];
	
	// Get all images from FastAPI response (ranked best to worst)
	$all_images = isset( $result['image_options'] ) && is_array( $result['image_options'] ) ? $result['image_options'] : array();
	
	// Filter images: pick first unused as featured, next 10 unused as alternates
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$featured_image = null;
	$alternate_images = array();
	
	foreach ( $all_images as $image ) {
		$pexels_id = isset( $image['pexels_id'] ) ? strval( $image['pexels_id'] ) : '';
		
		// Skip if this image has been used before
		if ( empty( $pexels_id ) || in_array( $pexels_id, $used_pexels_ids, true ) ) {
			continue;
		}
		
		// First unused image becomes featured image
		if ( $featured_image === null ) {
			$featured_image = $image;
			error_log( "RPM: Selected image #{$pexels_id} as featured (rank #" . (array_search($image, $all_images) + 1) . ")" );
		}
		// Next 10 unused images become alternates
		elseif ( count( $alternate_images ) < 10 ) {
			$alternate_images[] = $image;
		}
		
		// Stop after we have 1 featured + 10 alternates
		if ( count( $alternate_images ) >= 10 ) {
			break;
		}
	}
	
	error_log( "RPM: Image selection complete - Featured: " . ($featured_image ? $featured_image['pexels_id'] : 'none') . ", Alternates: " . count($alternate_images) );

	// Create WordPress post
	$post_data = array(
		'post_title'   => $generated_content['title'],
		'post_content' => $generated_content['content'],
		'post_excerpt' => $generated_content['excerpt'],
		'post_status'  => isset( $snapshot['post_status'] ) ? $snapshot['post_status'] : 'draft',
		'post_type'    => 'post',
	);

	// Insert post
	$post_id = wp_insert_post( $post_data, true );

	if ( is_wp_error( $post_id ) ) {
		wp_send_json_error(
			array(
				'message' => 'Failed to create post: ' . $post_id->get_error_message(),
			),
			500
		);
	}

	// Set categories
	if ( ! empty( $snapshot['categories'] ) && is_array( $snapshot['categories'] ) ) {
		wp_set_post_categories( $post_id, $snapshot['categories'] );
	}

	// Set tags - USE AI-GENERATED TAGS from FastAPI response (English)
	// Create tags if they don't exist
	if ( ! empty( $generated_content['tags'] ) && is_array( $generated_content['tags'] ) ) {
		$tag_ids = array();
		foreach ( $generated_content['tags'] as $tag_name ) {
			// Check if tag exists
			$tag = get_term_by( 'name', $tag_name, 'post_tag' );
			if ( ! $tag ) {
				// Create new tag
				$new_tag = wp_insert_term( $tag_name, 'post_tag' );
				if ( ! is_wp_error( $new_tag ) ) {
					$tag_ids[] = (int) $new_tag['term_id'];
				}
			} else {
				$tag_ids[] = (int) $tag->term_id;
			}
		}
		// Set tags by ID
		if ( ! empty( $tag_ids ) ) {
			wp_set_post_tags( $post_id, $tag_ids, false );
		}
	}

	// Save meta description
	if ( ! empty( $generated_content['meta_description'] ) ) {
		update_post_meta( $post_id, '_yoast_wpseo_metadesc', $generated_content['meta_description'] );
	}

	// Mark as auto-posted
	update_post_meta( $post_id, '_rpm_auto_posted', '1' );
	update_post_meta( $post_id, '_rpm_topic', $topic );
	update_post_meta( $post_id, '_rpm_generated_at', current_time( 'mysql' ) );

	// Handle featured image
	$image_id = null;
	if ( $featured_image && ! empty( $featured_image['url'] ) ) {
		$image_id = rpm_download_and_attach_image(
			$featured_image['url'],
			$post_id,
			array(
				'alt_text'         => $featured_image['alt_text'],
				'photographer'     => isset( $featured_image['photographer'] ) ? $featured_image['photographer'] : '',
				'photographer_url' => isset( $featured_image['photographer_url'] ) ? $featured_image['photographer_url'] : '',
				'pexels_id'        => isset( $featured_image['pexels_id'] ) ? $featured_image['pexels_id'] : '',
			)
		);

		if ( ! is_wp_error( $image_id ) && $image_id ) {
			set_post_thumbnail( $post_id, $image_id );
		}
	}

	// Store alternate images as post meta for later selection
	if ( ! empty( $alternate_images ) ) {
		$image_count = count( $alternate_images );
		update_post_meta( $post_id, '_rpm_image_options', $alternate_images );
		error_log( "RPM: Stored {$image_count} alternate images for post {$post_id}" );
	} else {
		error_log( "RPM: No alternate images available for post {$post_id}" );
	}
	
	// Store SEO structured data
	if ( isset( $generated_content['seo'] ) && is_array( $generated_content['seo'] ) ) {
		$seo_data = $generated_content['seo'];
		if ( ! empty( $seo_data['focus_keyphrase'] ) ) {
			update_post_meta( $post_id, '_rpm_focus_keyphrase', sanitize_text_field( $seo_data['focus_keyphrase'] ) );
		}
		if ( isset( $seo_data['reading_time_minutes'] ) ) {
			update_post_meta( $post_id, '_rpm_reading_time', absint( $seo_data['reading_time_minutes'] ) );
		}
		if ( isset( $seo_data['word_count'] ) ) {
			update_post_meta( $post_id, '_rpm_word_count', absint( $seo_data['word_count'] ) );
		}
	}
	
	// Store FAQ schema data
	if ( isset( $generated_content['faq'] ) && is_array( $generated_content['faq'] ) && ! empty( $generated_content['faq'] ) ) {
		update_post_meta( $post_id, '_rpm_faq_schema', wp_json_encode( $generated_content['faq'] ) );
	}
	
	// Store HowTo schema data
	if ( isset( $generated_content['howto'] ) && is_array( $generated_content['howto'] ) ) {
		$howto_data = $generated_content['howto'];
		if ( ! empty( $howto_data['is_tutorial'] ) && ! empty( $howto_data['steps'] ) ) {
			update_post_meta( $post_id, '_rpm_howto_schema', wp_json_encode( $howto_data ) );
		}
	}
	
	// Get the tags that were assigned
	$assigned_tags = wp_get_post_tags( $post_id, array( 'fields' => 'names' ) );
	
	// Remove topic from queue after successful post creation
	$queue = get_option( 'rpm_topic_queue', array() );
	$topic_removed = false;
	
	// If topic_id provided, use it for faster removal, otherwise fallback to subject search
	if ( ! empty( $topic_id ) ) {
		foreach ( $queue as $key => $queue_topic ) {
			if ( is_array( $queue_topic ) && isset( $queue_topic['id'] ) && $queue_topic['id'] === $topic_id ) {
				unset( $queue[ $key ] );
				$topic_removed = true;
				error_log( "RPM: Removed topic by ID '{$topic_id}' from queue after creating post {$post_id}" );
				break;
			}
		}
	} else {
		// Fallback: search by subject (for backward compatibility)
		foreach ( $queue as $key => $queue_topic ) {
			if ( is_array( $queue_topic ) && isset( $queue_topic['subject'] ) && $queue_topic['subject'] === $topic ) {
				unset( $queue[ $key ] );
				$topic_removed = true;
				error_log( "RPM: Removed topic by subject '{$topic}' from queue after creating post {$post_id}" );
				break;
			}
		}
	}
	
	if ( $topic_removed ) {
		$queue = array_values( $queue ); // Re-index array
		update_option( 'rpm_topic_queue', $queue );
	}
	
	// Return success response (including prompt and image data for debugging)
	wp_send_json_success(
		array(
			'post_id'         => $post_id,
			'post_url'        => get_permalink( $post_id ),
			'title'           => $generated_content['title'],
			'tags'            => $assigned_tags,  // Return actual assigned tags
			'image_id'        => $image_id,
			'featured_image'  => $featured_image,  // Selected featured image
			'image_options'   => $alternate_images,  // Filtered alternate images
			'total_images_received' => count( $all_images ),  // Total from FastAPI
			'edit_url'        => get_edit_post_link( $post_id, 'raw' ),
			'prompt'          => isset( $result['prompt'] ) ? $result['prompt'] : null,  // Prompt is at root level
			'topic_removed'   => ( $key !== false ),  // Indicate if topic was removed from queue
		)
	);
}

add_action( 'wp_ajax_rpm_generate_single_post', 'rpm_ajax_generate_single_post' );

/**
 * AJAX handler to remove a topic from the queue
 */
function rpm_ajax_remove_topic() {
	// Verify nonce
	check_ajax_referer( 'rpm_remove_topic', 'nonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}

	// Get topic
	$topic = isset( $_POST['topic'] ) ? sanitize_text_field( $_POST['topic'] ) : '';

	if ( empty( $topic ) ) {
		wp_send_json_error(
			array(
				'message' => 'Topic is required',
			),
			400
		);
	}

	// Get current queue
	$queue = get_option( 'rpm_topic_queue', array() );

	// Remove topic
	$key = array_search( $topic, $queue, true );
	if ( $key !== false ) {
		unset( $queue[ $key ] );
		$queue = array_values( $queue ); // Re-index array
		update_option( 'rpm_topic_queue', $queue );

		wp_send_json_success(
			array(
				'message'    => 'Topic removed from queue',
				'remaining'  => count( $queue ),
			)
		);
	} else {
		wp_send_json_error(
			array(
				'message' => 'Topic not found in queue',
			),
			404
		);
	}
}

add_action( 'wp_ajax_rpm_remove_topic', 'rpm_ajax_remove_topic' );

/**
 * AJAX handler to test FastAPI connection
 */
function rpm_ajax_test_fastapi_connection() {
	// Verify nonce
	check_ajax_referer( 'rpm_test_fastapi', 'nonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}

	// Get FastAPI settings
	$fastapi_url = isset( $_POST['fastapi_url'] ) ? esc_url_raw( $_POST['fastapi_url'] ) : get_option( 'rpm_fastapi_url' );
	$fastapi_key = isset( $_POST['fastapi_key'] ) ? sanitize_text_field( $_POST['fastapi_key'] ) : get_option( 'rpm_fastapi_key' );

	if ( empty( $fastapi_url ) || empty( $fastapi_key ) ) {
		wp_send_json_error(
			array(
				'message' => 'FastAPI URL and API Key are required',
			),
			400
		);
	}

	// Test connection using PHP
	$response = wp_remote_get(
		rtrim( $fastapi_url, '/' ) . '/me',
		array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $fastapi_key,
				'Content-Type'  => 'application/json',
			),
			'timeout' => 10,
		)
	);

	// Check for errors
	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				'message' => 'Connection failed: ' . $response->get_error_message(),
			),
			500
		);
	}

	$response_code = wp_remote_retrieve_response_code( $response );
	$response_body = wp_remote_retrieve_body( $response );
	$data          = json_decode( $response_body, true );

	if ( $response_code !== 200 ) {
		$error_message = 'Connection failed with code ' . $response_code;
		if ( isset( $data['detail'] ) ) {
			$error_message = $data['detail'];
		}

		wp_send_json_error(
			array(
				'message' => $error_message,
			),
			$response_code
		);
	}

	// Success
	if ( isset( $data['success'] ) && $data['success'] && isset( $data['user'] ) ) {
		wp_send_json_success(
			array(
				'message' => 'Connection successful!',
				'user'    => array(
					'username' => $data['user']['username'],
					'email'    => $data['user']['email'],
					'permissions' => $data['user']['permissions'],
					'usage'    => isset( $data['user']['usage'] ) ? $data['user']['usage'] : null,
					'subscription' => isset( $data['user']['subscription'] ) ? $data['user']['subscription'] : null,
				),
			)
		);
	} else {
		wp_send_json_error(
			array(
				'message' => 'Unexpected response from FastAPI',
			),
			500
		);
	}
}

add_action( 'wp_ajax_rpm_test_fastapi_connection', 'rpm_ajax_test_fastapi_connection' );

/**
 * Sanitize snapshot data
 *
 * @param array $snapshot Raw snapshot data
 * @return array Sanitized snapshot
 */
function rpm_sanitize_snapshot( $snapshot ) {
	if ( ! is_array( $snapshot ) ) {
		return array();
	}

	$sanitized = array();

	// Writing parameters
	if ( isset( $snapshot['tone'] ) ) {
		$sanitized['tone'] = sanitize_text_field( $snapshot['tone'] );
	}

	if ( isset( $snapshot['writing_style'] ) ) {
		$sanitized['writing_style'] = sanitize_text_field( $snapshot['writing_style'] );
	}

	if ( isset( $snapshot['word_count'] ) ) {
		$sanitized['word_count'] = absint( $snapshot['word_count'] );
	}

	if ( isset( $snapshot['content_goal'] ) ) {
		$sanitized['content_goal'] = sanitize_text_field( $snapshot['content_goal'] );
	}

	if ( isset( $snapshot['expertise_level'] ) ) {
		$sanitized['expertise_level'] = sanitize_text_field( $snapshot['expertise_level'] );
	}

	// Site & Audience Context
	if ( isset( $snapshot['site_description'] ) ) {
		$sanitized['site_description'] = sanitize_textarea_field( $snapshot['site_description'] );
	}

	if ( isset( $snapshot['target_audience'] ) ) {
		$sanitized['target_audience'] = sanitize_textarea_field( $snapshot['target_audience'] );
	}

	if ( isset( $snapshot['industry'] ) ) {
		$sanitized['industry'] = sanitize_text_field( $snapshot['industry'] );
	}

	// SEO Keywords
	if ( isset( $snapshot['brand_keywords'] ) ) {
		$sanitized['brand_keywords'] = sanitize_textarea_field( $snapshot['brand_keywords'] );
	}

	if ( isset( $snapshot['avoid_words'] ) ) {
		$sanitized['avoid_words'] = sanitize_textarea_field( $snapshot['avoid_words'] );
	}

	if ( isset( $snapshot['seo_keywords'] ) && is_array( $snapshot['seo_keywords'] ) ) {
		$sanitized['seo_keywords'] = array_map( 'sanitize_text_field', $snapshot['seo_keywords'] );
	}

	// WordPress settings
	if ( isset( $snapshot['categories'] ) && is_array( $snapshot['categories'] ) ) {
		$sanitized['categories'] = array_map( 'absint', $snapshot['categories'] );
	}
	
	if ( isset( $snapshot['category_name'] ) ) {
		$sanitized['category_name'] = sanitize_text_field( $snapshot['category_name'] );
	}

	if ( isset( $snapshot['tags'] ) && is_array( $snapshot['tags'] ) ) {
		$sanitized['tags'] = array_map( 'sanitize_text_field', $snapshot['tags'] );
	}

	if ( isset( $snapshot['include_images'] ) ) {
		$sanitized['include_images'] = (bool) $snapshot['include_images'];
	}

	if ( isset( $snapshot['image_query'] ) ) {
		$sanitized['image_query'] = sanitize_text_field( $snapshot['image_query'] );
	}

	if ( isset( $snapshot['post_status'] ) ) {
		$allowed_statuses           = array( 'publish', 'draft', 'pending', 'private' );
		$sanitized['post_status'] = in_array( $snapshot['post_status'], $allowed_statuses, true ) ? $snapshot['post_status'] : 'draft';
	}

	return $sanitized;
}

/**
 * Download and attach image to post
 *
 * @param string $image_url Image URL
 * @param int    $post_id Post ID
 * @param array  $metadata Image metadata
 * @return int|WP_Error Attachment ID or error
 */
function rpm_download_and_attach_image( $image_url, $post_id, $metadata = array() ) {
	require_once ABSPATH . 'wp-admin/includes/media.php';
	require_once ABSPATH . 'wp-admin/includes/file.php';
	require_once ABSPATH . 'wp-admin/includes/image.php';

	// Download image
	$temp_file = download_url( $image_url, 30 );

	if ( is_wp_error( $temp_file ) ) {
		return $temp_file;
	}

	// Convert image to WebP format
	$file_name = basename( $image_url );
	
	// Load the image
	$image_data = file_get_contents( $temp_file );
	$image_resource = imagecreatefromstring( $image_data );
	
	if ( $image_resource === false ) {
		@unlink( $temp_file );
		return new WP_Error( 'image_conversion_failed', 'Failed to load image for conversion' );
	}
	
	// Create WebP version
	$webp_file = $temp_file . '.webp';
	$conversion_success = imagewebp( $image_resource, $webp_file, 85 );  // 85% quality
	imagedestroy( $image_resource );
	
	// Clean up original temp file
	@unlink( $temp_file );
	
	if ( ! $conversion_success ) {
		@unlink( $webp_file );
		return new WP_Error( 'webp_conversion_failed', 'Failed to convert image to WebP' );
	}
	
	// Prepare file for upload
	$file_name = preg_replace( '/\.(jpg|jpeg|png|gif)$/i', '.webp', $file_name );
	if ( ! preg_match( '/\.webp$/i', $file_name ) ) {
		$file_name .= '.webp';
	}
	
	$file_array = array(
		'name'     => $file_name,
		'tmp_name' => $webp_file,
		'type'     => 'image/webp',
	);

	// Upload WebP image to media library
	$attachment_id = media_handle_sideload( $file_array, $post_id );
	
	// Clean up WebP temp file
	@unlink( $webp_file );

	// Clean up temp file
	if ( is_wp_error( $attachment_id ) ) {
		@unlink( $temp_file );
		return $attachment_id;
	}

	// Set image metadata
	if ( ! empty( $metadata['alt_text'] ) ) {
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $metadata['alt_text'] ) );
	}

	if ( ! empty( $metadata['pexels_id'] ) ) {
		update_post_meta( $attachment_id, '_pexels_id', sanitize_text_field( $metadata['pexels_id'] ) );
	}

	if ( ! empty( $metadata['photographer'] ) ) {
		update_post_meta( $attachment_id, '_photographer', sanitize_text_field( $metadata['photographer'] ) );
	}

	if ( ! empty( $metadata['photographer_url'] ) ) {
		update_post_meta( $attachment_id, '_photographer_url', esc_url_raw( $metadata['photographer_url'] ) );
	}

	return $attachment_id;
}

/**
 * AJAX handler to analyze pages and intelligently fill form
 */
function rpm_ajax_ai_analyze_pages() {
	// Verify nonce
	check_ajax_referer( 'rpm_ai_analyze_pages', '_wpnonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions', 403 );
	}

	// Get page IDs
	$page_ids = isset( $_POST['page_ids'] ) ? array_map( 'intval', (array) $_POST['page_ids'] ) : array();
	
	if ( empty( $page_ids ) ) {
		wp_send_json_error( 'No pages selected', 400 );
	}

	// Gather page content
	$pages_content = array();
	foreach ( $page_ids as $page_id ) {
		$page = get_post( $page_id );
		if ( $page && $page->post_status === 'publish' ) {
			$pages_content[] = array(
				'title' => $page->post_title,
				'content' => wp_strip_all_tags( $page->post_content ),
				'excerpt' => $page->post_excerpt ? wp_strip_all_tags( $page->post_excerpt ) : ''
			);
		}
	}

	if ( empty( $pages_content ) ) {
		wp_send_json_error( 'No valid pages found', 400 );
	}

	// Get FastAPI settings (use same defaults as the rest of the plugin)
	$fastapi_url = get_option( 'rpm_fastapi_url', 'http://localhost:8000' );
	$fastapi_key = get_option( 'rpm_fastapi_key', '' );

	if ( empty( $fastapi_key ) ) {
		wp_send_json_error( 'FastAPI API key must be configured', 400 );
	}

	// Prepare request to FastAPI
	$endpoint = rtrim( $fastapi_url, '/' ) . '/analyze-site';
	
	$request_body = array(
		'pages' => $pages_content,
		'site_url' => get_site_url()
	);

	// Make request to FastAPI
	$response = wp_remote_post(
		$endpoint,
		array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $fastapi_key,
				'Content-Type'  => 'application/json',
			),
			'body'    => wp_json_encode( $request_body ),
			'timeout' => 60,
		)
	);

	if ( is_wp_error( $response ) ) {
		error_log( 'RPM: FastAPI analyze-site error: ' . $response->get_error_message() );
		wp_send_json_error( 'Failed to connect to AI service: ' . $response->get_error_message(), 500 );
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );

	if ( $status_code !== 200 ) {
		error_log( 'RPM: FastAPI analyze-site failed with status ' . $status_code . ': ' . $body );
		wp_send_json_error( 'AI analysis failed (HTTP ' . $status_code . ')', 500 );
	}

	$result = json_decode( $body, true );

	if ( json_last_error() !== JSON_ERROR_NONE ) {
		error_log( 'RPM: Failed to parse FastAPI analyze-site response: ' . json_last_error_msg() );
		wp_send_json_error( 'Invalid response from AI service', 500 );
	}

	// Return the analyzed data
	wp_send_json_success( $result );
}

add_action( 'wp_ajax_rpm_ai_analyze_pages', 'rpm_ajax_ai_analyze_pages' );

/**
 * AJAX handler to suggest tags based on page content
 */
function rpm_ajax_suggest_tags() {
	// Verify nonce
	if ( ! wp_verify_nonce( $_POST['_wpnonce'], 'rpm_suggest_tags' ) ) {
		wp_send_json_error( 'Invalid nonce', 403 );
	}

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions', 403 );
	}

	// Get parameters
	$page_ids = isset( $_POST['page_ids'] ) ? array_map( 'intval', (array) $_POST['page_ids'] ) : array();
	$count = isset( $_POST['count'] ) ? intval( $_POST['count'] ) : 10;
	
	// Validate count
	$count = max( 5, min( 50, $count ) );
	
	if ( empty( $page_ids ) ) {
		wp_send_json_error( 'No pages selected', 400 );
	}

	// Gather page content
	$pages = array();
	foreach ( $page_ids as $page_id ) {
		$post = get_post( $page_id );
		if ( $post && $post->post_status === 'publish' ) {
			$pages[] = array(
				'title' => $post->post_title,
				'content' => wp_strip_all_tags( $post->post_content ),
				'excerpt' => $post->post_excerpt ? wp_strip_all_tags( $post->post_excerpt ) : ''
			);
		}
	}

	if ( empty( $pages ) ) {
		wp_send_json_error( 'No valid pages found', 400 );
	}

	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();
	
	// Call FastAPI to suggest tags
	$result = $fastapi_client->suggest_tags( $pages, $count );

	// Check for errors
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message(), 500 );
	}

	// Return response
	if ( isset( $result['success'] ) && $result['success'] ) {
		wp_send_json_success( array(
			'tags' => isset( $result['tags'] ) ? $result['tags'] : array(),
			'message' => isset( $result['message'] ) ? $result['message'] : ''
		) );
	} else {
		wp_send_json_error( isset( $result['message'] ) ? $result['message'] : 'Failed to generate tag suggestions', 500 );
	}
}

add_action( 'wp_ajax_rpm_suggest_tags', 'rpm_ajax_suggest_tags' );

/**
 * AJAX handler to suggest categories based on page content
 */
function rpm_ajax_suggest_categories() {
	// Verify nonce
	if ( ! wp_verify_nonce( $_POST['_wpnonce'], 'rpm_suggest_categories' ) ) {
		wp_send_json_error( 'Invalid nonce', 403 );
	}

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions', 403 );
	}

	// Get parameters
	$page_ids = isset( $_POST['page_ids'] ) ? array_map( 'intval', (array) $_POST['page_ids'] ) : array();
	$count = isset( $_POST['count'] ) ? intval( $_POST['count'] ) : 5;
	
	// Validate count
	$count = max( 3, min( 20, $count ) );
	
	if ( empty( $page_ids ) ) {
		wp_send_json_error( 'No pages selected', 400 );
	}

	// Gather page content
	$pages = array();
	foreach ( $page_ids as $page_id ) {
		$post = get_post( $page_id );
		if ( $post && $post->post_status === 'publish' ) {
			$pages[] = array(
				'title' => $post->post_title,
				'content' => wp_strip_all_tags( $post->post_content ),
				'excerpt' => $post->post_excerpt ? wp_strip_all_tags( $post->post_excerpt ) : ''
			);
		}
	}

	if ( empty( $pages ) ) {
		wp_send_json_error( 'No valid pages found', 400 );
	}

	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();
	
	// Call FastAPI to suggest categories
	$result = $fastapi_client->suggest_categories( $pages, $count );

	// Check for errors
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message(), 500 );
	}

	// Return response
	if ( isset( $result['success'] ) && $result['success'] ) {
		wp_send_json_success( array(
			'categories' => isset( $result['categories'] ) ? $result['categories'] : array(),
			'message' => isset( $result['message'] ) ? $result['message'] : ''
		) );
	} else {
		wp_send_json_error( isset( $result['message'] ) ? $result['message'] : 'Failed to generate category suggestions', 500 );
	}
}

add_action( 'wp_ajax_rpm_suggest_categories', 'rpm_ajax_suggest_categories' );

/**
 * AJAX handler to generate multiple posts in batch
 */
function rpm_ajax_generate_batch_posts() {
	// Verify nonce
	check_ajax_referer( 'rpm_batch_generate', 'nonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}

	// Get request data
	$count = isset( $_POST['count'] ) ? absint( $_POST['count'] ) : 1;

	if ( $count < 1 ) {
		wp_send_json_error(
			array(
				'message' => 'Invalid count. Must be at least 1.',
			),
			400
		);
	}

	// Get topic queue
	$queue = get_option( 'rpm_topic_queue', array() );

	if ( empty( $queue ) ) {
		wp_send_json_error(
			array(
				'message' => 'No topics in queue',
			),
			400
		);
	}

	// Limit to available topics
	$count = min( $count, count( $queue ) );

	// Get existing tags for AI to choose from
	$existing_tags = get_tags( array( 'hide_empty' => false, 'orderby' => 'count', 'order' => 'DESC' ) );
	$tag_names = array();
	foreach ( $existing_tags as $tag ) {
		$tag_names[] = $tag->name;
	}

	// Get default settings for content generation
	$default_snapshot = array(
		'site_description'   => get_option( 'rpm_site_description', '' ),
		'industry'           => get_option( 'rpm_industry', '' ),
		'target_audience'    => get_option( 'rpm_target_audience', '' ),
		'tone'               => get_option( 'rpm_content_tone', 'professional' ),
		'writing_style'      => get_option( 'rpm_writing_style', 'balanced' ),
		'content_goal'       => get_option( 'rpm_content_goal', 'inform' ),
		'expertise_level'    => get_option( 'rpm_expertise_level', 'intermediate' ),
		'word_count'         => rpm_content_length_to_words( get_option( 'rpm_content_length', 'medium' ) ),
		'brand_keywords'     => get_option( 'rpm_brand_keywords', '' ),
		'avoid_words'        => get_option( 'rpm_avoid_words', '' ),
		'image_search'       => get_option( 'rpm_image_search', '' ),
		'post_status'        => get_option( 'rpm_default_status', 'draft' ),
		'include_images'     => true,
		'marketing_mode'     => get_option( 'rpm_marketing_mode', false ),
		'tags'               => $tag_names,  // Send existing tags for AI to prefer
	);

	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();

	$created = 0;
	$failed = 0;
	$errors = array();

	// Process topics
	for ( $i = 0; $i < $count; $i++ ) {
		if ( empty( $queue ) ) {
			break;
		}

		$topic_data = array_shift( $queue );
		$topic = isset( $topic_data['subject'] ) ? $topic_data['subject'] : ( is_string( $topic_data ) ? $topic_data : '' );

		if ( empty( $topic ) ) {
			$failed++;
			continue;
		}

		// Merge topic-specific settings with defaults
		$snapshot = $default_snapshot;
		
		// Set category if specified in topic
		if ( ! empty( $topic_data['category_id'] ) ) {
			$snapshot['categories'] = array( absint( $topic_data['category_id'] ) );
			$category = get_category( $topic_data['category_id'] );
			if ( $category ) {
				$snapshot['category_name'] = $category->name;
			}
		}

		// Merge content profile if exists
		if ( ! empty( $topic_data['content_profile'] ) && is_array( $topic_data['content_profile'] ) ) {
			$snapshot = array_merge( $snapshot, rpm_sanitize_snapshot( $topic_data['content_profile'] ) );
		}

		// Add excluded Pexels IDs to prevent re-using images
		$snapshot['exclude_pexels_ids'] = rpm_get_used_pexels_ids();

		// Generate content
		$result = $fastapi_client->generate_content( $topic, $snapshot );

		if ( is_wp_error( $result ) ) {
			$failed++;
			$errors[] = 'Topic "' . $topic . '": ' . $result->get_error_message();
			continue;
		}

		if ( ! isset( $result['success'] ) || ! $result['success'] ) {
			$failed++;
			$errors[] = 'Topic "' . $topic . '": Failed to generate content';
			continue;
		}

		$generated_content = $result['generated_content'];
		
		// Get images
		$all_images = isset( $result['image_options'] ) && is_array( $result['image_options'] ) ? $result['image_options'] : array();
		$used_pexels_ids = rpm_get_used_pexels_ids();
		$featured_image = null;
		$alternate_images = array();
		
		foreach ( $all_images as $image ) {
			$pexels_id = isset( $image['pexels_id'] ) ? strval( $image['pexels_id'] ) : '';
			
			if ( empty( $pexels_id ) || in_array( $pexels_id, $used_pexels_ids, true ) ) {
				continue;
			}
			
			if ( $featured_image === null ) {
				$featured_image = $image;
			} elseif ( count( $alternate_images ) < 10 ) {
				$alternate_images[] = $image;
			}
			
			if ( count( $alternate_images ) >= 10 ) {
				break;
			}
		}

		// Create WordPress post
		$post_data = array(
			'post_title'   => $generated_content['title'],
			'post_content' => $generated_content['content'],
			'post_excerpt' => $generated_content['excerpt'],
			'post_status'  => $snapshot['post_status'],
			'post_type'    => 'post',
		);

		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			$failed++;
			$errors[] = 'Topic "' . $topic . '": ' . $post_id->get_error_message();
			continue;
		}

		// Set categories
		if ( ! empty( $snapshot['categories'] ) && is_array( $snapshot['categories'] ) ) {
			wp_set_post_categories( $post_id, $snapshot['categories'] );
		}

		// Set AI-generated tags
		if ( ! empty( $generated_content['tags'] ) && is_array( $generated_content['tags'] ) ) {
			$tag_ids = array();
			foreach ( $generated_content['tags'] as $tag_name ) {
				$tag = get_term_by( 'name', $tag_name, 'post_tag' );
				if ( ! $tag ) {
					$new_tag = wp_insert_term( $tag_name, 'post_tag' );
					if ( ! is_wp_error( $new_tag ) ) {
						$tag_ids[] = (int) $new_tag['term_id'];
					}
				} else {
					$tag_ids[] = (int) $tag->term_id;
				}
			}
			if ( ! empty( $tag_ids ) ) {
				wp_set_post_tags( $post_id, $tag_ids, false );
			}
		}

		// Save meta description
		if ( ! empty( $generated_content['meta_description'] ) ) {
			update_post_meta( $post_id, '_yoast_wpseo_metadesc', $generated_content['meta_description'] );
		}

		// Mark as auto-posted
		update_post_meta( $post_id, '_rpm_auto_posted', '1' );
		update_post_meta( $post_id, '_rpm_topic', $topic );
		update_post_meta( $post_id, '_rpm_generated_at', current_time( 'mysql' ) );

		// Handle featured image
		if ( $featured_image && ! empty( $featured_image['url'] ) ) {
			$image_id = rpm_download_and_attach_image(
				$featured_image['url'],
				$post_id,
				array(
					'alt_text'         => $featured_image['alt_text'],
					'photographer'     => isset( $featured_image['photographer'] ) ? $featured_image['photographer'] : '',
					'photographer_url' => isset( $featured_image['photographer_url'] ) ? $featured_image['photographer_url'] : '',
					'pexels_id'        => isset( $featured_image['pexels_id'] ) ? $featured_image['pexels_id'] : '',
				)
			);

			if ( ! is_wp_error( $image_id ) && $image_id ) {
				set_post_thumbnail( $post_id, $image_id );
			}
		}

		// Store alternate images
		if ( ! empty( $alternate_images ) ) {
			update_post_meta( $post_id, '_rpm_image_options', $alternate_images );
		}

		// Store SEO structured data
		if ( isset( $generated_content['seo'] ) && is_array( $generated_content['seo'] ) ) {
			$seo_data = $generated_content['seo'];
			if ( ! empty( $seo_data['focus_keyphrase'] ) ) {
				update_post_meta( $post_id, '_rpm_focus_keyphrase', sanitize_text_field( $seo_data['focus_keyphrase'] ) );
			}
			if ( isset( $seo_data['reading_time_minutes'] ) ) {
				update_post_meta( $post_id, '_rpm_reading_time', absint( $seo_data['reading_time_minutes'] ) );
			}
			if ( isset( $seo_data['word_count'] ) ) {
				update_post_meta( $post_id, '_rpm_word_count', absint( $seo_data['word_count'] ) );
			}
		}
		
		// Store FAQ schema data
		if ( isset( $generated_content['faq'] ) && is_array( $generated_content['faq'] ) && ! empty( $generated_content['faq'] ) ) {
			update_post_meta( $post_id, '_rpm_faq_schema', wp_json_encode( $generated_content['faq'] ) );
		}
		
		// Store HowTo schema data
		if ( isset( $generated_content['howto'] ) && is_array( $generated_content['howto'] ) ) {
			$howto_data = $generated_content['howto'];
			if ( ! empty( $howto_data['is_tutorial'] ) && ! empty( $howto_data['steps'] ) ) {
				update_post_meta( $post_id, '_rpm_howto_schema', wp_json_encode( $howto_data ) );
			}
		}

		$created++;
	}

	// Update queue
	update_option( 'rpm_topic_queue', $queue );

	// Return results
	wp_send_json_success(
		array(
			'created' => $created,
			'failed'  => $failed,
			'errors'  => $errors,
		)
	);
}

add_action( 'wp_ajax_rpm_generate_batch_posts', 'rpm_ajax_generate_batch_posts' );

/**
 * AJAX handler to generate a SINGLE post from the queue (for progress tracking)
 */
function rpm_ajax_generate_single_post_from_queue() {
	// Verify nonce
	check_ajax_referer( 'rpm_batch_generate', 'nonce' );

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}

	// Get topic queue
	$queue = get_option( 'rpm_topic_queue', array() );

	if ( empty( $queue ) ) {
		wp_send_json_error(
			array(
				'message' => 'No topics in queue',
			),
			400
		);
	}

	// Get first topic from queue
	$topic_data = array_shift( $queue );
	$topic = isset( $topic_data['subject'] ) ? $topic_data['subject'] : ( is_string( $topic_data ) ? $topic_data : '' );

	if ( empty( $topic ) ) {
		// Update queue (remove empty topic) and fail
		update_option( 'rpm_topic_queue', $queue );
		wp_send_json_error(
			array(
				'message' => 'Invalid topic in queue',
			),
			400
		);
	}

	// Get existing tags for AI to choose from
	$existing_tags = get_tags( array( 'hide_empty' => false, 'orderby' => 'count', 'order' => 'DESC' ) );
	$tag_names = array();
	foreach ( $existing_tags as $tag ) {
		$tag_names[] = $tag->name;
	}

	// Get default settings
	$default_snapshot = array(
		'site_description'   => get_option( 'rpm_site_description', '' ),
		'industry'           => get_option( 'rpm_industry', '' ),
		'target_audience'    => get_option( 'rpm_target_audience', '' ),
		'tone'               => get_option( 'rpm_content_tone', 'professional' ),
		'writing_style'      => get_option( 'rpm_writing_style', 'balanced' ),
		'content_goal'       => get_option( 'rpm_content_goal', 'inform' ),
		'expertise_level'    => get_option( 'rpm_expertise_level', 'intermediate' ),
		'word_count'         => rpm_content_length_to_words( get_option( 'rpm_content_length', 'medium' ) ),
		'brand_keywords'     => get_option( 'rpm_brand_keywords', '' ),
		'avoid_words'        => get_option( 'rpm_avoid_words', '' ),
		'image_search'       => get_option( 'rpm_image_search', '' ),
		'post_status'        => get_option( 'rpm_default_status', 'draft' ),
		'include_images'     => true,
		'marketing_mode'     => get_option( 'rpm_marketing_mode', false ),
		'tags'               => $tag_names,  // Send existing tags for AI to prefer
	);

	$snapshot = $default_snapshot;
	
	// Set category if specified in topic
	if ( ! empty( $topic_data['category_id'] ) ) {
		$snapshot['categories'] = array( absint( $topic_data['category_id'] ) );
		$category = get_category( $topic_data['category_id'] );
		if ( $category ) {
			$snapshot['category_name'] = $category->name;
		}
	}

	// Merge content profile if exists
	if ( ! empty( $topic_data['content_profile'] ) && is_array( $topic_data['content_profile'] ) ) {
		$snapshot = array_merge( $snapshot, rpm_sanitize_snapshot( $topic_data['content_profile'] ) );
	}

	// Add excluded Pexels IDs to prevent re-using images
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$snapshot['exclude_pexels_ids'] = $used_pexels_ids;
	error_log( "RPM Single Post Gen - Sending " . count( $used_pexels_ids ) . " excluded Pexels IDs to API" );

	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();

	// Generate content
	$result = $fastapi_client->generate_content( $topic, $snapshot );

	if ( is_wp_error( $result ) ) {
		// Don't remove from queue on error
		array_unshift( $queue, $topic_data );
		update_option( 'rpm_topic_queue', $queue );
		
		wp_send_json_error(
			array(
				'message' => $result->get_error_message(),
				'topic'   => $topic,
			),
			500
		);
	}

	if ( ! isset( $result['success'] ) || ! $result['success'] ) {
		// Don't remove from queue on error
		array_unshift( $queue, $topic_data );
		update_option( 'rpm_topic_queue', $queue );
		
		wp_send_json_error(
			array(
				'message' => 'Failed to generate content',
				'topic'   => $topic,
			),
			500
		);
	}

	$generated_content = $result['generated_content'];
	
	// DEBUG: Log the entire API response structure
	error_log( "RPM DEBUG - Full API response keys: " . implode( ', ', array_keys( $result ) ) );
	if ( isset( $result['image_options'] ) ) {
		error_log( "RPM DEBUG - image_options is SET, type: " . gettype( $result['image_options'] ) . ", count: " . ( is_array( $result['image_options'] ) ? count( $result['image_options'] ) : 'N/A' ) );
		if ( is_array( $result['image_options'] ) && count( $result['image_options'] ) > 0 ) {
			error_log( "RPM DEBUG - First image keys: " . implode( ', ', array_keys( $result['image_options'][0] ) ) );
		}
	} else {
		error_log( "RPM DEBUG - image_options is NOT SET in response" );
	}
	
	// Get images
	$all_images = isset( $result['image_options'] ) && is_array( $result['image_options'] ) ? $result['image_options'] : array();
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$featured_image = null;
	$alternate_images = array();
	
	error_log( "RPM DEBUG - all_images count: " . count( $all_images ) );
	error_log( "RPM DEBUG - Used pexels IDs: " . implode( ', ', $used_pexels_ids ) );
	
	foreach ( $all_images as $index => $image ) {
		$pexels_id = isset( $image['pexels_id'] ) ? strval( $image['pexels_id'] ) : '';
		
		if ( empty( $pexels_id ) || in_array( $pexels_id, $used_pexels_ids, true ) ) {
			error_log( "RPM DEBUG - Skipping image $index (pexels_id: $pexels_id) - " . ( empty( $pexels_id ) ? 'empty ID' : 'already used' ) );
			continue;
		}
		
		if ( $featured_image === null ) {
			$featured_image = $image;
			error_log( "RPM DEBUG - Set image $index as featured (pexels_id: $pexels_id)" );
		} elseif ( count( $alternate_images ) < 10 ) {
			$alternate_images[] = $image;
			error_log( "RPM DEBUG - Added image $index as alternate #" . count( $alternate_images ) . " (pexels_id: $pexels_id)" );
		}
		
		if ( count( $alternate_images ) >= 10 ) {
			error_log( "RPM DEBUG - Reached 10 alternates, stopping" );
			break;
		}
	}
	
	error_log( "RPM DEBUG - Final: featured=" . ( $featured_image ? 'YES' : 'NO' ) . ", alternates=" . count( $alternate_images ) );

	// Create WordPress post
	$post_data = array(
		'post_title'   => $generated_content['title'],
		'post_content' => $generated_content['content'],
		'post_excerpt' => $generated_content['excerpt'],
		'post_status'  => $snapshot['post_status'],
		'post_type'    => 'post',
	);

	$post_id = wp_insert_post( $post_data, true );

	if ( is_wp_error( $post_id ) ) {
		// Don't remove from queue on error
		array_unshift( $queue, $topic_data );
		update_option( 'rpm_topic_queue', $queue );
		
		wp_send_json_error(
			array(
				'message' => 'Failed to create WordPress post: ' . $post_id->get_error_message(),
				'topic'   => $topic,
			),
			500
		);
	}

	// Set category
	if ( ! empty( $snapshot['categories'] ) ) {
		wp_set_post_categories( $post_id, $snapshot['categories'] );
	}

	// Mark as auto-posted
	update_post_meta( $post_id, '_rpm_auto_posted', '1' );
	update_post_meta( $post_id, '_rpm_topic', $topic );
	update_post_meta( $post_id, '_rpm_generated_at', current_time( 'mysql' ) );

	// Handle featured image
	if ( $featured_image ) {
		$image_url = isset( $featured_image['url'] ) ? $featured_image['url'] : '';
		$photographer = isset( $featured_image['photographer'] ) ? $featured_image['photographer'] : '';
		$pexels_url = isset( $featured_image['photographer_url'] ) ? $featured_image['photographer_url'] : '';
		$pexels_id = isset( $featured_image['pexels_id'] ) ? $featured_image['pexels_id'] : '';

		error_log( "RPM Single Post Gen - Post {$post_id}: Image URL = {$image_url}" );

		if ( ! empty( $image_url ) ) {
			$attachment_id = rpm_download_and_attach_image(
				$image_url,
				$post_id,
				array(
					'alt_text'         => isset( $featured_image['alt_text'] ) ? $featured_image['alt_text'] : '',
					'photographer'     => $photographer,
					'photographer_url' => $pexels_url,
					'pexels_id'        => $pexels_id,
				)
			);

			if ( is_wp_error( $attachment_id ) ) {
				error_log( "RPM Single Post Gen - Post {$post_id}: Image download FAILED: " . $attachment_id->get_error_message() );
			} elseif ( $attachment_id ) {
				error_log( "RPM Single Post Gen - Post {$post_id}: Image attached successfully (ID: {$attachment_id})" );
				set_post_thumbnail( $post_id, $attachment_id );
				
				// Track used Pexels ID
				if ( ! empty( $pexels_id ) ) {
					rpm_add_used_pexels_id( $pexels_id );
				}
			} else {
				error_log( "RPM Single Post Gen - Post {$post_id}: Image attachment returned empty/false" );
			}
		} else {
			error_log( "RPM Single Post Gen - Post {$post_id}: No image URL provided" );
		}
	} else {
		error_log( "RPM Single Post Gen - Post {$post_id}: No featured_image data" );
	}

	// Store alternate images in post meta
	error_log( "RPM Single Post Gen - Post {$post_id}: Alternate images count: " . count( $alternate_images ) );
	if ( ! empty( $alternate_images ) ) {
		update_post_meta( $post_id, '_rpm_image_options', $alternate_images );
		error_log( "RPM Single Post Gen - Post {$post_id}: Stored " . count( $alternate_images ) . " alternate images" );
	} else {
		error_log( "RPM Single Post Gen - Post {$post_id}: No alternate images to store (array is empty)" );
	}

	// Store SEO structured data
	if ( isset( $generated_content['seo'] ) && is_array( $generated_content['seo'] ) ) {
		$seo_data = $generated_content['seo'];
		if ( ! empty( $seo_data['focus_keyphrase'] ) ) {
			update_post_meta( $post_id, '_rpm_focus_keyphrase', sanitize_text_field( $seo_data['focus_keyphrase'] ) );
		}
		if ( isset( $seo_data['reading_time_minutes'] ) ) {
			update_post_meta( $post_id, '_rpm_reading_time', absint( $seo_data['reading_time_minutes'] ) );
		}
		if ( isset( $seo_data['word_count'] ) ) {
			update_post_meta( $post_id, '_rpm_word_count', absint( $seo_data['word_count'] ) );
		}
	}
	
	// Store FAQ schema data
	if ( isset( $generated_content['faq'] ) && is_array( $generated_content['faq'] ) && ! empty( $generated_content['faq'] ) ) {
		update_post_meta( $post_id, '_rpm_faq_schema', wp_json_encode( $generated_content['faq'] ) );
	}
	
	// Store HowTo schema data
	if ( isset( $generated_content['howto'] ) && is_array( $generated_content['howto'] ) ) {
		$howto_data = $generated_content['howto'];
		if ( ! empty( $howto_data['is_tutorial'] ) && ! empty( $howto_data['steps'] ) ) {
			update_post_meta( $post_id, '_rpm_howto_schema', wp_json_encode( $howto_data ) );
		}
	}

	// Assign tags
	if ( ! empty( $generated_content['tags'] ) && is_array( $generated_content['tags'] ) ) {
		$tag_ids = array();
		foreach ( $generated_content['tags'] as $tag_name ) {
			$tag = get_term_by( 'name', $tag_name, 'post_tag' );
			if ( ! $tag ) {
				$new_tag = wp_insert_term( $tag_name, 'post_tag' );
				if ( ! is_wp_error( $new_tag ) ) {
					$tag_ids[] = (int) $new_tag['term_id'];
				}
			} else {
				$tag_ids[] = (int) $tag->term_id;
			}
		}
		if ( ! empty( $tag_ids ) ) {
			wp_set_post_tags( $post_id, $tag_ids, false );
			error_log( "RPM Single Post Gen - Post {$post_id}: Assigned " . count( $tag_ids ) . " tags" );
		}
	} else {
		error_log( "RPM Single Post Gen - Post {$post_id}: No tags in generated content" );
	}

	// SUCCESS - Update queue (topic already removed by array_shift)
	update_option( 'rpm_topic_queue', $queue );

	wp_send_json_success(
		array(
			'post_id'        => $post_id,
			'post_title'     => $generated_content['title'],
			'post_url'       => get_edit_post_link( $post_id, 'raw' ),
			'remaining'      => count( $queue ),
			'topic'          => $topic,
		)
	);
}

add_action( 'wp_ajax_rpm_generate_single_post_from_queue', 'rpm_ajax_generate_single_post_from_queue' );

/**
 * Helper function to convert content length to word count
 */
function rpm_content_length_to_words( $length ) {
	$word_counts = array(
		'short'  => 500,
		'medium' => 1000,
		'long'   => 1500,
	);

	return isset( $word_counts[ $length ] ) ? $word_counts[ $length ] : 1000;
}

/**
 * AJAX handler to verify setup code via server-side
 */
function rpm_ajax_verify_setup_code() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'rpm_verify_setup_code' ) ) {
		wp_send_json_error( array( 'message' => 'Invalid nonce' ), 403 );
	}

	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
	}

	// Get setup code
	$code = isset( $_POST['code'] ) ? sanitize_text_field( $_POST['code'] ) : '';

	if ( empty( $code ) || strlen( $code ) !== 6 ) {
		wp_send_json_error( array( 'message' => 'Invalid code format' ), 400 );
	}

	// Get FastAPI URL (allow override or use default)
	$fastapi_url = isset( $_POST['fastapi_url'] ) ? esc_url_raw( $_POST['fastapi_url'] ) : 'https://findexa.ca';

	// Make request to verify code
	$response = wp_remote_post(
		rtrim( $fastapi_url, '/' ) . '/api/setup/verify-code',
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode( array( 'code' => $code ) ),
			'timeout' => 15,
		)
	);

	// Check for errors
	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array( 'message' => 'Connection failed: ' . $response->get_error_message() ),
			500
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( $status_code !== 200 ) {
		$error_message = 'Verification failed';
		if ( isset( $data['detail'] ) ) {
			$error_message = $data['detail'];
		} elseif ( isset( $data['message'] ) ) {
			$error_message = $data['message'];
		}

		wp_send_json_error( array( 'message' => $error_message ), $status_code );
	}

	// Success - return the configuration
	wp_send_json_success( $data );
}

add_action( 'wp_ajax_rpm_verify_setup_code', 'rpm_ajax_verify_setup_code' );

/**
 * Helper function to generate a single topic using AI
 * 
 * @param array $snapshot Content profile settings
 * @return string|WP_Error Generated topic or error
 */
function rpm_generate_single_topic_ai( $snapshot ) {
	// Use FastAPI client to generate topic
	// ALL OpenAI calls must go through FastAPI, never directly from WordPress
	$fastapi_client = new RPM_FastAPI_Client();
	$result = $fastapi_client->generate_topic( $snapshot );
	
	if ( is_wp_error( $result ) ) {
		error_log( "RPM Topic Generation Error: " . $result->get_error_message() );
		return $result;
	}
	
	// Extract topic from response
	if ( ! isset( $result['topic'] ) || empty( $result['topic'] ) ) {
		return new WP_Error( 'no_topic', 'No topic returned from FastAPI' );
	}
	
	$topic = trim( $result['topic'] );
	error_log( "RPM: Generated topic via FastAPI: $topic" );
	
	return $topic;
}

/**
 * AJAX handler to generate a post on-the-fly (without using queue)
 */
function rpm_ajax_generate_post_on_the_fly() {
	// Verify nonce
	check_ajax_referer( 'rpm_on_the_fly', 'nonce' );
	
	// Check user permissions
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array(
				'message' => 'Insufficient permissions',
			),
			403
		);
	}
	
	// Get existing tags for AI to choose from
	$existing_tags = get_tags( array( 'hide_empty' => false, 'orderby' => 'count', 'order' => 'DESC' ) );
	$tag_names = array();
	foreach ( $existing_tags as $tag ) {
		$tag_names[] = $tag->name;
	}
	
	// Get default settings
	$default_snapshot = array(
		'site_description'   => get_option( 'rpm_site_description', '' ),
		'industry'           => get_option( 'rpm_industry', '' ),
		'target_audience'    => get_option( 'rpm_target_audience', '' ),
		'tone'               => get_option( 'rpm_content_tone', 'professional' ),
		'writing_style'      => get_option( 'rpm_writing_style', 'balanced' ),
		'content_goal'       => get_option( 'rpm_content_goal', 'inform' ),
		'expertise_level'    => get_option( 'rpm_expertise_level', 'intermediate' ),
		'word_count'         => rpm_content_length_to_words( get_option( 'rpm_content_length', 'medium' ) ),
		'brand_keywords'     => get_option( 'rpm_brand_keywords', '' ),
		'avoid_words'        => get_option( 'rpm_avoid_words', '' ),
		'image_search'       => get_option( 'rpm_image_search', '' ),
		'post_status'        => get_option( 'rpm_default_status', 'draft' ),
		'include_images'     => true,
		'marketing_mode'     => get_option( 'rpm_marketing_mode', false ),
		'tags'               => $tag_names,
	);
	
	$snapshot = $default_snapshot;
	
	// Get default category if set, otherwise pick a random one
	$default_category_id = get_option( 'rpm_default_category', '' );
	if ( ! empty( $default_category_id ) ) {
		$snapshot['categories'] = array( absint( $default_category_id ) );
		$category = get_category( $default_category_id );
		if ( $category ) {
			$snapshot['category_name'] = $category->name;
		}
	} else {
		// No category selected - pick a random category
		$all_categories = get_categories( array(
			'hide_empty' => false,
			'exclude'    => array( 1 ), // Exclude "Uncategorized" (ID 1)
		) );
		
		if ( ! empty( $all_categories ) ) {
			$random_category = $all_categories[ array_rand( $all_categories ) ];
			$snapshot['categories'] = array( $random_category->term_id );
			$snapshot['category_name'] = $random_category->name;
			error_log( "RPM On-the-Fly - No category set, randomly selected: " . $random_category->name );
		}
	}
	
	// Add excluded Pexels IDs to prevent re-using images
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$snapshot['exclude_pexels_ids'] = $used_pexels_ids;
	error_log( "RPM On-the-Fly - Sending " . count( $used_pexels_ids ) . " excluded Pexels IDs to API" );
	
	// Generate a single topic using AI
	$topic = rpm_generate_single_topic_ai( $snapshot );
	
	if ( is_wp_error( $topic ) ) {
		wp_send_json_error(
			array(
				'message' => 'Failed to generate topic: ' . $topic->get_error_message(),
			),
			500
		);
	}
	
	error_log( "RPM On-the-Fly - Generated topic: $topic" );
	
	// Initialize FastAPI client
	$fastapi_client = new RPM_FastAPI_Client();
	
	// Check if this is from Daily Auto-Post trigger (vs Manual Batch Processing)
	$is_daily_auto_post = isset( $_POST['is_daily_auto_post'] ) && $_POST['is_daily_auto_post'] === 'true';
	
	// Generate content
	$result = $fastapi_client->generate_content( $topic, $snapshot, $is_daily_auto_post );
	
	if ( is_wp_error( $result ) ) {
		wp_send_json_error(
			array(
				'message' => $result->get_error_message(),
				'topic'   => $topic,
			),
			500
		);
	}
	
	if ( ! isset( $result['success'] ) || ! $result['success'] ) {
		wp_send_json_error(
			array(
				'message' => 'Failed to generate content',
				'topic'   => $topic,
			),
			500
		);
	}
	
	$generated_content = $result['generated_content'];
	
	// Get images
	$all_images = isset( $result['image_options'] ) && is_array( $result['image_options'] ) ? $result['image_options'] : array();
	$used_pexels_ids = rpm_get_used_pexels_ids();
	$featured_image = null;
	$alternate_images = array();
	
	foreach ( $all_images as $index => $image ) {
		$pexels_id = isset( $image['pexels_id'] ) ? strval( $image['pexels_id'] ) : '';
		
		if ( empty( $pexels_id ) || in_array( $pexels_id, $used_pexels_ids, true ) ) {
			continue;
		}
		
		if ( $featured_image === null ) {
			$featured_image = $image;
		} elseif ( count( $alternate_images ) < 10 ) {
			$alternate_images[] = $image;
		}
		
		if ( count( $alternate_images ) >= 10 ) {
			break;
		}
	}
	
	// Create WordPress post
	$post_data = array(
		'post_title'   => $generated_content['title'],
		'post_content' => $generated_content['content'],
		'post_excerpt' => $generated_content['excerpt'],
		'post_status'  => $snapshot['post_status'],
		'post_type'    => 'post',
	);
	
	$post_id = wp_insert_post( $post_data, true );
	
	if ( is_wp_error( $post_id ) ) {
		wp_send_json_error(
			array(
				'message' => 'Failed to create WordPress post: ' . $post_id->get_error_message(),
				'topic'   => $topic,
			),
			500
		);
	}
	
	// Set category
	if ( ! empty( $snapshot['categories'] ) ) {
		wp_set_post_categories( $post_id, $snapshot['categories'] );
	}
	
	// Mark as auto-posted
	update_post_meta( $post_id, '_rpm_auto_posted', '1' );
	update_post_meta( $post_id, '_rpm_topic', $topic );
	update_post_meta( $post_id, '_rpm_generated_at', current_time( 'mysql' ) );
	
	// Handle featured image
	if ( $featured_image ) {
		$image_url = isset( $featured_image['url'] ) ? $featured_image['url'] : '';
		$photographer = isset( $featured_image['photographer'] ) ? $featured_image['photographer'] : '';
		$pexels_url = isset( $featured_image['photographer_url'] ) ? $featured_image['photographer_url'] : '';
		$pexels_id = isset( $featured_image['pexels_id'] ) ? $featured_image['pexels_id'] : '';
		
		if ( ! empty( $image_url ) ) {
			$attachment_id = rpm_download_and_attach_image(
				$image_url,
				$post_id,
				array(
					'alt_text'         => isset( $featured_image['alt_text'] ) ? $featured_image['alt_text'] : '',
					'photographer'     => $photographer,
					'photographer_url' => $pexels_url,
					'pexels_id'        => $pexels_id,
				)
			);
			
			if ( ! is_wp_error( $attachment_id ) && $attachment_id ) {
				set_post_thumbnail( $post_id, $attachment_id );
				
				// Track used Pexels ID
				if ( ! empty( $pexels_id ) ) {
					rpm_add_used_pexels_id( $pexels_id );
				}
			}
		}
	}
	
	// Store alternate images in post meta
	if ( ! empty( $alternate_images ) ) {
		update_post_meta( $post_id, '_rpm_image_options', $alternate_images );
	}
	
	// Store SEO structured data
	if ( isset( $generated_content['seo'] ) && is_array( $generated_content['seo'] ) ) {
		$seo_data = $generated_content['seo'];
		if ( ! empty( $seo_data['focus_keyphrase'] ) ) {
			update_post_meta( $post_id, '_rpm_focus_keyphrase', sanitize_text_field( $seo_data['focus_keyphrase'] ) );
		}
		if ( isset( $seo_data['reading_time_minutes'] ) ) {
			update_post_meta( $post_id, '_rpm_reading_time', absint( $seo_data['reading_time_minutes'] ) );
		}
		if ( isset( $seo_data['word_count'] ) ) {
			update_post_meta( $post_id, '_rpm_word_count', absint( $seo_data['word_count'] ) );
		}
	}
	
	// Store FAQ schema data
	if ( isset( $generated_content['faq'] ) && is_array( $generated_content['faq'] ) && ! empty( $generated_content['faq'] ) ) {
		update_post_meta( $post_id, '_rpm_faq_schema', wp_json_encode( $generated_content['faq'] ) );
	}
	
	// Store HowTo schema data
	if ( isset( $generated_content['howto'] ) && is_array( $generated_content['howto'] ) ) {
		$howto_data = $generated_content['howto'];
		if ( ! empty( $howto_data['is_tutorial'] ) && ! empty( $howto_data['steps'] ) ) {
			update_post_meta( $post_id, '_rpm_howto_schema', wp_json_encode( $howto_data ) );
		}
	}
	
	// Assign tags
	if ( ! empty( $generated_content['tags'] ) && is_array( $generated_content['tags'] ) ) {
		$tag_ids = array();
		foreach ( $generated_content['tags'] as $tag_name ) {
			$tag = get_term_by( 'name', $tag_name, 'post_tag' );
			if ( ! $tag ) {
				$new_tag = wp_insert_term( $tag_name, 'post_tag' );
				if ( ! is_wp_error( $new_tag ) ) {
					$tag_ids[] = (int) $new_tag['term_id'];
				}
			} else {
				$tag_ids[] = (int) $tag->term_id;
			}
		}
		if ( ! empty( $tag_ids ) ) {
			wp_set_post_tags( $post_id, $tag_ids, false );
		}
	}
	
	// SUCCESS
	wp_send_json_success(
		array(
			'post_id'        => $post_id,
			'post_title'     => $generated_content['title'],
			'post_url'       => get_edit_post_link( $post_id, 'raw' ),
			'topic'          => $topic,
		)
	);
}

add_action( 'wp_ajax_rpm_generate_post_on_the_fly', 'rpm_ajax_generate_post_on_the_fly' );
